import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { DEBOUNCE_TIMEOUT } from 'constants/constants';

import { InputSize } from 'shared/consts/InputSize';
import { InputSearch } from 'shared/ui/InputSearch/InputSearch';
import { Spin } from 'shared/ui/Spin/Spin';

import NoSearchResultView from 'components/GlobalSidebar/ModalSearchView/NoSearchResultsView';
import { GLOBAL_SEARCH_REQUESTS, REQUESTS } from 'components/GlobalSidebar/ModalSearchView/request';
import SearchErrorView from 'components/GlobalSidebar/ModalSearchView/SearchErrorView';
import SearchResults from 'components/GlobalSidebar/ModalSearchView/SearchResults';
import ShortSearchValueView from 'components/GlobalSidebar/ModalSearchView/ShortSearchValueView';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSearchView/index.i18n';

import style from 'components/GlobalSidebar/ModalSearchView/index.css';

const MIN_SEARCH_LENGTH = 3;
const SEARCH_LIMIT = 50;
const SHORT_SEARCH_REQUEST_ADVICE_TIMEOUT = 3000;

export interface ISearchCar {
    id: string;
    model_id: string;
    number: string;
    vin: string;
}

const ModalSearchView = () => {
    const [searchValue, setSearchValue] = useState<string>('');
    const [isSearching, setIsSearching] = useState<boolean>(false);
    const [searchingError, setSearchingError] = useState<Error | null>(null);
    const [searchResultCars, setSearchResultCars] = useState<ISearchCar[] | null>(null);
    const request: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: GLOBAL_SEARCH_REQUESTS }),
    );

    const debounceTimeout: MutableRefObject<any> = useRef(null);

    const shortSearchAdviceTimeout: MutableRefObject<any> = useRef(null);
    const [isSearchRequestShort, setIsSearchRequestShort] = useState<boolean>(false);

    useEffect(() => {
        shortSearchAdviceTimeout.current = setTimeout(() => {
            setIsSearchRequestShort(true);
        }, SHORT_SEARCH_REQUEST_ADVICE_TIMEOUT);

        return () => {
            clearTimeout(debounceTimeout.current);
            clearTimeout(shortSearchAdviceTimeout.current);
        };
    }, []);

    useEffect(() => {
        clearTimeout(debounceTimeout.current);
        debounceTimeout.current = setTimeout(() => {
            search();
        }, DEBOUNCE_TIMEOUT);
    }, [searchValue]);

    const formatSearchValue = React.useMemo(() => searchValue.trim().replace(/\s+/gi, '_'), [searchValue]);

    const search = () => {
        if (searchValue?.length >= MIN_SEARCH_LENGTH) {
            setIsSearchRequestShort(true);
            setIsSearching(!!setSearchResultCars?.length);
            setSearchingError(null);
            request.current.abort();
            request.current
                .exec(REQUESTS.SEARCH_CARS, {
                    queryParams: {
                        limit: SEARCH_LIMIT,
                        has_all_of: formatSearchValue,
                        what: 'cars',
                    },
                })
                .then((response) => {
                    if (response?.meta !== ABORT_ERROR_KEY) {
                        setIsSearching(false);
                        let cars = response?.objects?.cars ?? [];
                        setSearchResultCars(cars);
                    }
                })
                .catch((error) => {
                    setIsSearching(false);
                    setSearchingError(error);
                });
        } else {
            setIsSearching(false);
            setSearchingError(null);
            request.current.abort();
            setSearchResultCars(null);
        }
    };

    return (
        <div className={style.modal_search_view}>
            <div className={style.search_input_container}>
                <InputSearch
                    inputSize={InputSize.M}
                    placeholder={i18n('Search by number, VIN or model')}
                    value={searchValue}
                    hasAutoFocus
                    onInputChange={setSearchValue}
                />
            </div>
            {searchingError ? (
                <SearchErrorView reloadFunction={search} />
            ) : isSearching && !searchResultCars?.length ? (
                <div className={style.controls_container}>
                    <Spin />
                </div>
            ) : searchResultCars && !searchResultCars?.length ? (
                <NoSearchResultView />
            ) : searchResultCars ? (
                <SearchResults
                    isSearching={isSearching}
                    searchResults={searchResultCars}
                />
            ) : searchValue?.length < MIN_SEARCH_LENGTH && isSearchRequestShort ? (
                <ShortSearchValueView />
            ) : null}
        </div>
    );
};

export default ModalSearchView;
