import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import shortDateTime from 'utils/date/shortDateTime';

import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import Empty from 'components/GlobalSidebar/ModalSessionView/SessionPhotos/empty';
import PhotoShimmer from 'components/GlobalSidebar/ModalSessionView/SessionPhotos/photoShimmer';
import { SectionPhotoRowItem } from 'components/GlobalSidebar/ModalSessionView/SessionPhotos/sectionPhotosRow';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import Gallery from 'components/ui/Gallery';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSessionView/SessionPhotos/index.i18n';

import styles from 'components/GlobalSidebar/ModalSessionView/SessionPhotos/index.css';

interface ImgItem {
    url: string;
    preview_url: string;
    created_at: number;
}

interface resultImgObject {
    size: number;
    images: ImgItem[];
    groups: Array<{ data: ImgItem[] }>;
}

const SessionPhotos = React.memo(({ session_id }: { session_id: string | null }) => {
    let [isLoading, setIsLoading] = useState(true);
    let [response, setResponse] = useState<null | resultImgObject>(null);
    let [currentImageItem, setCurrentImageItem] = useState<number | null>(null);
    let [currentImageLabel, setCurrentImageLabel] = useState<string>('');
    let request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));
    let { notificationDispatch } = React.useContext(NotificationCenterContext) || {};

    useEffect(() => {
        if (session_id) {
            request.current
                .exec(REQUESTS.GET_PHOTOS, {
                    queryParams: {
                        session_id,
                    },
                })
                .then((response) => {
                    // Group photos by dates
                    let groups: Record<string, { data: ImgItem[] }> =
                        response?.images?.reduce((memo, item) => {
                            const roundedTime = Math.round(item.created_at / 100) * 100;

                            if (!memo.hasOwnProperty(roundedTime)) {
                                memo[roundedTime] = { data: [item] };
                            } else {
                                memo[roundedTime].data.push(item);
                            }

                            return memo;
                        }, {}) || {};

                    setResponse({
                        size: response?.images?.length,
                        images: response?.images,
                        groups: Object.values(groups),
                    });

                    setIsLoading(false);
                })
                .catch((error) => {
                    let notificationOptions = {
                        title: i18n('Bad Gateway'),
                        iconType: NotificationIconType.ERROR,
                    };

                    if (error.status == '403') {
                        notificationOptions.title = i18n('Permission denied');
                    }

                    notificationDispatch(addNotification(notificationOptions));
                    setResponse(null);
                    setIsLoading(false);
                });
        }

        return () => {
            request.current.abort();
        };
    }, []);

    const onSetImageLabel = React.useCallback(() => {
        let img: ImgItem | undefined = currentImageItem !== null ? response?.images?.[currentImageItem] : undefined;
        let date = img?.created_at ? shortDateTime(img?.created_at) : '';

        setCurrentImageLabel(date);
    }, [currentImageItem, response]);

    const onImageClickHandler = React.useCallback((ration, index) => {
        setCurrentImageItem(ration + index);
    }, []);

    useEffect(() => {
        onSetImageLabel();
    }, [currentImageItem, onSetImageLabel]);

    const onNextClickHandler = React.useCallback(() => {
        if (response?.size && currentImageItem !== null) {
            if (currentImageItem < response.size - 1) {
                setCurrentImageItem(currentImageItem + 1);
            } else {
                setCurrentImageItem(0);
            }
        }
    }, [currentImageItem, response]);

    const onPrevClickHandler = React.useCallback(() => {
        if (response?.size && currentImageItem !== null) {
            if (currentImageItem > 0) {
                setCurrentImageItem(currentImageItem - 1);
            } else {
                setCurrentImageItem(response.size - 1);
            }
        }
    }, [currentImageItem, response]);

    return (
        <div className={styles.container}>
            {isLoading ? (
                <PhotoShimmer />
            ) : !response?.size ? (
                <Empty />
            ) : (
                response?.groups?.map(({ data }, index) => {
                    let ration = 0;

                    if (index !== 0 && response) {
                        for (let i = 0; i < index; i++) {
                            ration = ration + response.groups[index - i - 1].data.length;
                        }
                    }

                    return (
                        <SectionPhotoRowItem
                            data={data}
                            onClick={onImageClickHandler.bind(null, ration)}
                            key={index}
                        />
                    );
                })
            )}

            {currentImageItem !== null && (
                <Gallery
                    label={currentImageLabel}
                    curIndex={currentImageItem}
                    maxIndex={response?.size || 0}
                    url={response?.images?.[currentImageItem]?.url || ''}
                    next={onNextClickHandler}
                    prev={onPrevClickHandler}
                    onClose={setCurrentImageItem.bind(null, null)}
                />
            )}
        </div>
    );
});

export default SessionPhotos;
