import React, { useEffect, useState } from 'react';

import { EMPTY_DATA, ONE_SECOND } from 'constants/constants';

import { getI18nLocale } from 'utils/language/getI18nLocale';
import { getLang } from 'utils/language/getLang';
import initMapApi from 'utils/map/map/initMapApi';
import generateId from 'utils/map/trace/generateId';
import { IconTraceType } from 'utils/map/trace/types';

import { getAddress } from 'entities/Geo/helpers/getAddress/getAddress';

import { Spin } from 'shared/ui/Spin/Spin';

import { PinIconTraceLayout } from 'components/Map/controls/PinIconTraceLayout';
import { IViolationItem } from 'components/Map/types';

import { i18n } from 'components/GlobalSidebar/ModalSessionView/TrackTimeline/index.i18n';

import style from 'components/GlobalSidebar/ModalSessionView/TrackTimeline/index.css';

export interface ITimelineItem {
    type: IconTraceType;
    overrideType: IconTraceType;
    coords: [number | null, number | null];
    timestamp: number | null;
    translated?: string;
    meta?: {
        violation?: IViolationItem;
    };
}

export interface ITracksForDisplay {
    type: IconTraceType;
    overrideType?: IconTraceType;
    coords: [number | null, number | null];
    meta: string | null;
    time: string | null;
    address: string | null;
    timestamp: number | null;
    is_hard?: boolean;
}

interface ITrackTimelineProps {
    timeline: ITimelineItem[];
    isLoading: boolean;
    hovered: string | null;
    hover: any;
    onClick: any;
}

const TrackTimeline = (props: ITrackTimelineProps) => {
    let { timeline, isLoading, hover, hovered, onClick } = props;
    const [addressLoading, setAddressLoading] = useState<boolean>(false);
    const [displayTracks, setDisplayTracks] = useState<ITracksForDisplay[]>([]);
    let mapScript = initMapApi();
    const lang = getLang();
    const locale = getI18nLocale();

    const mouseEnter = (id) => {
        hover(id);
    };
    const mouseLeave = () => {
        hover(null);
    };

    useEffect(() => {
        let displayTracks: ITracksForDisplay[] = timeline.map((track) => {
            let result: ITracksForDisplay = {
                type: track.type,
                overrideType: track.overrideType,
                coords: track.coords,
                meta: null,
                time: null,
                address: null,
                timestamp: track.timestamp,
                is_hard: false,
            };

            switch (track.type) {
                case IconTraceType.CAMERA:
                    result.meta = track.translated!;
                    break;
                case IconTraceType.START:
                    result.meta = i18n('Ride start');
                    break;
                case IconTraceType.FINISH:
                    result.meta = i18n('Ride finish');
                    break;
                case IconTraceType.CURRENT:
                    result.meta = i18n('Current car position');
                    break;
                case IconTraceType.SPEEDING:
                    let { peak = 0, limit = 0, is_hard } = track.meta?.violation || {};
                    let diff = Math.ceil(peak - limit);
                    result.meta = i18n('Speeding: +{value} km/h', {
                        value: diff,
                    });

                    result.is_hard = is_hard;
                    break;
                case IconTraceType.ACCELERATION:
                    result.meta = i18n('Harsh acceleration');
                    break;
                case IconTraceType.BRAKING:
                    result.meta = i18n('Harsh braking');
                    break;
                case IconTraceType.STRAIGHT_LATERAL_ACCELERATION:
                    result.meta = i18n('Harsh lane change');
                    break;
                case IconTraceType.TURNING_LATERAL_ACCELERATION:
                    result.meta = i18n('Harsh turn');
                    break;
                default:
                    result.meta = i18n('Dangerous maneuver');
                    result.is_hard = true;
                    break;
            }

            const isTrackTimestampValid = (timestamp) => new Date(timestamp).getTime() > 0;
            result.time = isTrackTimestampValid(track.timestamp)
                ? new Date((track?.timestamp ?? 0) * ONE_SECOND).toLocaleTimeString(locale, {
                      hour: '2-digit',
                      minute: '2-digit',
                  })
                : EMPTY_DATA;

            return result;
        });

        setDisplayTracks(displayTracks);

        if (displayTracks.some((displayTrack) => displayTrack.coords.every((coords) => coords))) {
            setAddressLoading(true);

            const setAddresses = () => {
                getAddresses(displayTracks.map((track) => track.coords))
                    .finally(() => {
                        setAddressLoading(false);
                    })
                    .then((addresses) => {
                        let newDisplayTracks = displayTracks.map((displayTrack, index) => {
                            return Object.assign({}, displayTrack, {
                                address: addresses[index],
                            });
                        });
                        setDisplayTracks(newDisplayTracks);
                    })
                    .catch(() => {});
            };

            // @ts-ignore
            if (window.ymaps) {
                setAddresses();
            } else {
                mapScript.addEventListener('load', () => {
                    setAddresses();
                });
            }
        }
    }, [timeline]);

    const getAddresses = (coordsArray: [number | null, number | null][]) => {
        const promises = coordsArray.map((coords) => {
            return getAddress(coords).catch(() => null);
        });

        return Promise.all(promises);
    };

    return (
        <div className={`${style.timeline} ${style[lang]}`}>
            {isLoading ? (
                <div className={style.spin_container}>
                    <Spin />
                </div>
            ) : (
                displayTracks.map((track, index, arr) => {
                    let { type, overrideType, meta, time, address, timestamp, is_hard } = track ?? {};
                    let data: any = Object.assign({}, track);
                    data.center = track.coords;
                    let id = generateId(data);
                    if (overrideType) {
                        type = overrideType;
                    }

                    return (
                        <React.Fragment key={`${index}_${timestamp}_${meta}_${address}`}>
                            <div
                                className={`${style.timeline_item} ${id === hovered ? style.hovered : ''}`}
                                onClick={onClick.bind(null, id)}
                                onMouseEnter={mouseEnter.bind(null, id)}
                                onMouseLeave={mouseLeave}
                            >
                                <div className={style.time}>{time}</div>
                                <div className={style.pin}>
                                    <PinIconTraceLayout
                                        type={type}
                                        options={{ is_hard, value: '' }}
                                    />
                                </div>
                                <div className={style.track_info}>
                                    <div className={style.track_meta}>{meta}</div>
                                    {addressLoading ? (
                                        <div className={style.address_shimmer} />
                                    ) : (
                                        <div
                                            className={style.address}
                                            title={address ?? EMPTY_DATA}
                                        >
                                            {address ?? EMPTY_DATA}
                                        </div>
                                    )}
                                </div>
                            </div>
                            {index !== arr.length - 1 ? <div className={style.track_connector} /> : null}
                        </React.Fragment>
                    );
                })
            )}
        </div>
    );
};

export default TrackTimeline;
