import React, { lazy, MutableRefObject, useContext, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_BACK, MODAL_OBJECT_SESSION_TAB } from 'constants/constants';

import FlagsContext from 'contexts/FlagsContext';

import { isShowAcceptancePhotos } from 'utils/isShowAcceptancePhotos';
import { IconTraceType } from 'utils/map/trace/types';
import getCurrent from 'utils/sessions/getCurrent';
import getFinish from 'utils/sessions/getFinish';
import getStart from 'utils/sessions/getStart';
import { ISession, ISessionTrackFeaturesPropertiesViolation } from 'utils/sessions/types';

import { getCarSignalDescriptionDisplayName } from 'entities/Car/helpers/getCarSignalDescriptionDisplayName/getCarSignalDescriptionDisplayName';

import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import { ErrorHandler } from 'components/GlobalErrorHandler';
import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import SessionPhotos from 'components/GlobalSidebar/ModalSessionView/SessionPhotos';
import TrackTimeline, { ITimelineItem } from 'components/GlobalSidebar/ModalSessionView/TrackTimeline';
import { REQUESTS as SIGNALS_REQUESTS_ENUM, SIGNALS_REQUESTS } from 'components/Signals/request';
import Tabs from 'components/ui/Tabs';
import { TabsSize } from 'components/ui/Tabs/types';
import Header2 from 'components/ui/Text/Header2';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSessionView/index.i18n';

import style from 'components/GlobalSidebar/ModalSessionView/index.css';

interface IModalSessionViewProps {
    sessionId: string | null;
}

enum TabsItemEnum {
    events = 'events',
    photos = 'photos',
}

const TrackMap = lazy(() => import('components/Map/TrackMap'));

const ModalSessionView = ({ sessionId }: IModalSessionViewProps) => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));
    const requestSignals: MutableRefObject<RequestHelper> = useRef(
        new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }),
    );
    const [session, setSession] = useState<ISession | null>(null);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [timeline, setTimeline] = useState<ITimelineItem[]>([]);
    const flags = useContext(FlagsContext);

    const [hovered, hover] = useState(null);
    const [clicked, onClick] = useState(null);
    const [currentTab, setCurrentTab] = useState<string>(TabsItemEnum.events);
    let location = useLocation();
    let history = useHistory();

    const TabsItem: Record<TabsItemEnum, string> = {
        [TabsItemEnum.events]: i18n('Events'),
        [TabsItemEnum.photos]: i18n('Photos'),
    };

    useEffect(() => {
        return () => {
            request.current.abort();
        };
    }, []);

    const getData = () => {
        setIsLoading(true);
        getSession().then((session: ISession) => {
            let trackPointsInfo: {
                start;
                aggressions;
                camera;
                finish;
                current;
                violations;
            } = {
                start: null,
                aggressions: [],
                camera: [],
                finish: null,
                current: null,
                violations: [],
            };

            trackPointsInfo.start = getStart(session);
            trackPointsInfo.finish = getFinish(session);
            trackPointsInfo.current = getCurrent(session);

            let violations: ISessionTrackFeaturesPropertiesViolation[] = session?.track?.features?.reduce?.(
                (_p: ISessionTrackFeaturesPropertiesViolation[], _c) => {
                    let violation = _c?.properties?.violation;
                    if (violation) {
                        violation.translated = i18n('{value} km/h', {
                            value: violation.peak.toFixed(0),
                        });

                        _p.push(violation);
                    }

                    return _p;
                },
                [],
            );

            trackPointsInfo.violations = violations;

            trackPointsInfo.aggressions = session?.aggressive_events || [];

            trackPointsInfo.camera = session?.camera_events || [];

            setSession(session);

            let tracksForDisplay = getTimeline(trackPointsInfo);
            setTimeline(tracksForDisplay);

            setIsLoading(false);
        });
    };

    useEffect(() => {
        getData();
    }, [sessionId]);

    useEffect(() => {
        updateTabs();
    }, [location.search]);

    const updateTabs = () => {
        let searchParams = new URLSearchParams(location?.search);
        let tab = searchParams.get(MODAL_OBJECT_SESSION_TAB) || TabsItemEnum.events;
        setCurrentTab(tab);
    };

    const onTabChange = (tabName) => {
        let searchParams = new URLSearchParams(location?.search);
        searchParams.set(MODAL_OBJECT_SESSION_TAB, tabName);

        history.push(`${location.pathname}?${searchParams}`);
    };

    const getSession = () => {
        return Promise.all([
            request.current.exec(REQUESTS.GET_CAR_SESSIONS, {
                queryParams: { session_id: sessionId, traits: 'ReportTrack' },
            }),
            requestSignals.current.exec(SIGNALS_REQUESTS_ENUM.GET_SIGNALS_DESCRIPTIONS),
        ]).then((response) => {
            let [sessionResponse, descriptionResponse] = response;

            const currentSession = sessionResponse?.sessions?.[0];

            if (currentSession && currentSession.camera_events) {
                currentSession.camera_events = currentSession.camera_events.map((event) => {
                    return {
                        ...event,
                        translated: getCarSignalDescriptionDisplayName(
                            descriptionResponse.signals_descriptions[event.kind],
                        ),
                    };
                });
            }

            return sessionResponse?.sessions?.[0] ?? null;
        });
    };

    const getTimeline = ({ start, aggressions, camera, finish, violations, current }): ITimelineItem[] => {
        let timeline = [
            {
                type: IconTraceType.START,
                coords: [start?.longitude ?? null, start?.latitude ?? null],
                timestamp: start?.timestamp ?? null,
            },

            ...(camera.map((event) => {
                return {
                    translated: event.translated,
                    type: IconTraceType.CAMERA,
                    coords: event.center || [null, null],
                    timestamp: event.ts,
                };
            }) ?? []),

            ...(aggressions?.map?.((event) => {
                let type: IconTraceType = IconTraceType.AGGRESSION;
                switch (event?.kind) {
                    case 'acceleration':
                        type = IconTraceType.ACCELERATION;
                        break;
                    case 'braking':
                        type = IconTraceType.BRAKING;
                        break;
                    case 'straight_lateral_acceleration':
                        type = IconTraceType.STRAIGHT_LATERAL_ACCELERATION;
                        break;
                    case 'turning_lateral_acceleration':
                        type = IconTraceType.TURNING_LATERAL_ACCELERATION;
                        break;
                }

                return {
                    type,
                    coords: event.center,
                    timestamp: event?.ts ?? null,
                };
            }) ?? []),
            ...(violations?.map?.((violation) => {
                return {
                    type: IconTraceType.SPEEDING,
                    overrideType: IconTraceType.EVENT,
                    coords: violation.center,
                    timestamp: violation?.start_time ?? null,
                    meta: {
                        violation,
                    },
                };
            }) ?? []),
        ]
            .filter((track) => track)
            .sort((a, b) => a.timestamp - b.timestamp);

        if (finish) {
            timeline.push({
                type: IconTraceType.FINISH,
                coords: [finish?.longitude ?? null, finish?.latitude ?? null],
                timestamp: finish?.timestamp ?? null,
            });
        }
        if (current) {
            timeline.push({
                type: IconTraceType.CURRENT,
                coords: [current?.longitude ?? null, current?.latitude ?? null],
                timestamp: current?.timestamp ?? null,
            });
        }

        return timeline;
    };

    const onBackClick = () => {
        let searchParams = new URLSearchParams(location.search);
        let backUrl = searchParams.get(MODAL_BACK);
        if (backUrl) {
            history.push(backUrl);
        }
    };

    return (
        <>
            <GlobalSidebarHeader backAction={onBackClick}>
                {!isLoading && (
                    <div>
                        <Header2>
                            <SessionTime
                                oneRow
                                start={getStart(session)?.timestamp}
                                finish={getFinish(session)?.timestamp}
                            />
                        </Header2>
                        <div className={style.label}> {i18n('Ride')}</div>
                    </div>
                )}
            </GlobalSidebarHeader>
            <div className={style.session_content}>
                <>
                    <div className={style.map_container}>
                        <ErrorHandler>
                            <React.Suspense fallback={<div />}>
                                <TrackMap
                                    timeline={timeline}
                                    session={session}
                                    hovered={hovered}
                                    hover={hover}
                                    clicked={clicked}
                                    options={{
                                        hideZoomControls: true,
                                        ...(flags?.center && {
                                            center: flags.center,
                                        }),
                                    }}
                                />
                            </React.Suspense>
                        </ErrorHandler>
                    </div>
                    <div className={style.session_tabs_content}>
                        {isShowAcceptancePhotos() && (
                            <div className={style.session_tabs_header}>
                                <Tabs
                                    size={TabsSize.M}
                                    tight
                                    activeTab={currentTab}
                                    onChange={onTabChange}
                                    tabs={Object.entries(TabsItem).map((el) => {
                                        return {
                                            link: el[0],
                                            name: el[1],
                                        };
                                    })}
                                />
                            </div>
                        )}

                        <div>
                            {currentTab === TabsItemEnum.events || !isShowAcceptancePhotos() ? (
                                <TrackTimeline
                                    isLoading={isLoading}
                                    hovered={hovered}
                                    hover={hover}
                                    onClick={onClick}
                                    timeline={timeline}
                                />
                            ) : currentTab && sessionId && isShowAcceptancePhotos() ? (
                                <SessionPhotos session_id={sessionId} />
                            ) : null}
                        </div>
                    </div>
                </>
            </div>
        </>
    );
};

export default ModalSessionView;
