import React, { MutableRefObject, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI, ONE_SECOND } from 'constants/constants';

import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import { ISignalDescriptionValue } from 'components/Signals/types';
import Confirm from 'components/ui/Confirm';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSignalSettings/DeleteSignalConfirm/index.i18n';

interface IDeleteGroupConfirmProps {
    onClose: () => void;
    signalDescription: ISignalDescriptionValue | null;
}

const DeleteSignalConfirm = ({ onClose, signalDescription }: IDeleteGroupConfirmProps) => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);

    let location = useLocation();
    let history = useHistory();

    const onDeleteClick = () => {
        setIsLoading(true);
        setError(null);
        request.current
            .exec(REQUESTS.DELETE_SIGNAL, {
                body: { signal_name: signalDescription?.name },
            })
            .then(() => {
                setTimeout(() => {
                    setIsLoading(false);

                    onClose();

                    let searchParams = new URLSearchParams(location.search);
                    searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                    searchParams.delete(MODAL_OBJECT_ID_CGI);
                    history.push(`${location.pathname}?${searchParams}`);
                }, ONE_SECOND);
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });
    };

    return (
        <Confirm
            title={i18n('Are you sure you want to delete the signal?')}
            description={i18n(
                'If you delete the signal, its history will disappear as well and you will not be able to recreate it with the same name',
            )}
            confirmButtonTitle={i18n('Yes, delete')}
            closeButtonTitle={i18n('Cancel')}
            onConfirm={onDeleteClick}
            onClose={onClose}
            isLoading={isLoading}
            error={error}
        />
    );
};

export default DeleteSignalConfirm;
