import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, ONE_SECOND } from 'constants/constants';

import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import DeleteSignalConfirm from 'components/GlobalSidebar/ModalSignalSettings/DeleteSignalConfirm';
import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import { ISignalDescriptionValue } from 'components/Signals/types';
import CommandButton from 'components/ui/Buttons/CommandButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import { Switcher } from 'components/ui/Switcher';
import Header2 from 'components/ui/Text/Header2';
import Header4 from 'components/ui/Text/Header4';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSignalSettings/index.i18n';

import coreStyle from 'components/Content/index.css';
import style from 'components/GlobalSidebar/ModalSignalSettings/index.css';

const ModalSignalSettings = () => {
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [signalDescription, setSignalDescription] = useState<ISignalDescriptionValue | null>(null);

    const [isChangingActive, setIsChangingActive] = useState<boolean>(false);
    const [changingActiveError, setChangingActiveError] = useState<Error | null>(null);
    const [isActive, setIsActive] = useState<boolean>(false);

    const [isDeleteConfirmOpen, setIsDeleteConfirmOpen] = useState<boolean>(false);

    let location = useLocation();

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));

    useEffect(() => {
        getDescriptions(true);

        return () => {
            request.current.abort();
        };
    }, []);

    const getDescriptions = (isInitial?: boolean) => {
        if (isInitial) {
            setIsLoading(true);
        }
        setError(null);
        request.current
            .exec(REQUESTS.GET_SIGNALS_DESCRIPTIONS)
            .then((response) => {
                let urlSearchParams = new URLSearchParams(location?.search);
                let signalName = urlSearchParams.get(MODAL_OBJECT_ID_CGI);

                setIsLoading(false);
                setError(null);
                let signalsDescriptions = response?.signals_descriptions ?? [];
                let signal: ISignalDescriptionValue | null = signalsDescriptions?.[signalName ?? ''] ?? null;

                setSignalDescription(signal);
                setIsActive(signal?.is_enabled ?? false);
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });
    };

    const onChangingActive = () => {
        setIsChangingActive(true);
        setChangingActiveError(null);
        request.current
            .exec(REQUESTS.CHANGE_SIGNALS_ACTIVE, {
                body: { signal_name: signalDescription?.name },
            })
            .then(() => {
                //Delay for backend to update info about signal before reloading
                setTimeout(() => {
                    getDescriptions();
                    setIsChangingActive(false);
                }, ONE_SECOND);
            })
            .catch((error) => {
                setIsChangingActive(false);
                setChangingActiveError(error);
            });
    };

    const onDeleteConfirmOpen = () => {
        setIsDeleteConfirmOpen(true);
    };

    const onDeleteConfirmClose = () => {
        setIsDeleteConfirmOpen(false);
    };

    return (
        <>
            <GlobalSidebarHeader>
                <div className={style.header}>
                    {isLoading ? (
                        <>
                            <div className={`${coreStyle.shimmer} ${style.title_shimmer}`} />
                            <div className={`${coreStyle.shimmer} ${style.description_shimmer}`} />
                        </>
                    ) : (
                        <>
                            <Header2>{signalDescription?.display_name ?? EMPTY_DATA}</Header2>
                            <span className={style.description}>{i18n('Signal')}</span>
                        </>
                    )}
                </div>
            </GlobalSidebarHeader>
            {error ? (
                <div className={style.error_container}>
                    <ErrorReloadLabel reloadFunction={getDescriptions} />
                </div>
            ) : (
                <div className={`${style.content} ${isLoading ? style.loading : ''}`}>
                    <Header4>{i18n('Signal settings')}</Header4>

                    <div className={style.separator} />
                    <CommandButton
                        title={i18n('Turn on the signal')}
                        description={i18n(
                            'If you disable the signal, then events of this type will not be shown in the signal center',
                        )}
                        disabled={isLoading || isChangingActive}
                        control={
                            <Switcher
                                disabled={isLoading || isChangingActive}
                                onChange={onChangingActive}
                                checked={isActive}
                            />
                        }
                    />

                    {changingActiveError ? (
                        <div className={style.small_error_label}>
                            <ErrorLabel simple />
                        </div>
                    ) : null}
                    <div className={style.separator} />
                    <CommandButton
                        critical
                        title={i18n('Delete signal')}
                        description={i18n(
                            'If you delete the signal, its history will disappear as well and you will not be able to recreate it with the same name',
                        )}
                        disabled={isLoading || isChangingActive}
                        onClick={onDeleteConfirmOpen}
                    />
                </div>
            )}

            {isDeleteConfirmOpen ? (
                <DeleteSignalConfirm
                    onClose={onDeleteConfirmClose}
                    signalDescription={signalDescription}
                />
            ) : null}
        </>
    );
};

export default ModalSignalSettings;
