import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { ICarInfo } from 'utils/car/types';
import shortDateTime from 'utils/date/shortDateTime';

import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { CarNumberLabel } from 'components/Cars/CarNumberLabel';
import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import SignalEntityBlock from 'components/GlobalSidebar/ModalSignalView/SignalEntityBlock';
import { getSignalInfoFromString, ISignalInfo } from 'components/GlobalSidebar/ModalSignalView/utils';
import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import { FormattedSignal, ISignalsResponse } from 'components/Signals/types';
import { ModalObjectTypes } from 'components/types';
import ErrorLabel from 'components/ui/ErrorLabel';
import { Status } from 'components/ui/Status';
import Header2 from 'components/ui/Text/Header2';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalSignalView/index.i18n';

import style from 'components/GlobalSidebar/ModalSignalView/index.css';

interface IModalSignalViewProps {
    signalInfo: string | null;
}

const TRANSLATES = {
    'Блокировка машины': i18n('Car is blocked'),
    'Лаг телематики': i18n('Telematics lag'),
    'Выезд из зоны': i18n('Leaving geofence'),
};

const ModalSignalView = ({ signalInfo: signalInfoCGI }: IModalSignalViewProps) => {
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [formattedSignal, setFormattedSignal] = useState<FormattedSignal | null>(null);
    const [car, setCar] = useState<ICarInfo | null>(null);
    let location = useLocation();

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));
    const signalInfo: ISignalInfo | null = getSignalInfoFromString(signalInfoCGI);

    useEffect(() => {
        return () => {
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        if (signalInfo) {
            getSignal();
        }
    }, [signalInfoCGI]);

    const getSignal = () => {
        setIsLoading(true);
        setError(null);
        request.current.abort();

        let { since } = signalInfo ?? {};

        request.current
            .exec(REQUESTS.GET_SIGNALS, {
                queryParams: {
                    since: since ?? null,
                    until: since ?? null,
                },
            })
            .then((signalsResponse: ISignalsResponse) => {
                let { signals, signals_descriptions } = signalsResponse;
                let formattedSignal = signals
                    ? signals
                          ?.filter((signal) => {
                              let { linked_entities } = signal;
                              //if car or trace don't exist, it's must be right
                              let isRightSignal = !(signalInfo?.car || signalInfo?.trace);
                              if (isRightSignal) {
                                  return isRightSignal;
                              } else {
                                  let isRightCar = true;
                                  if (signalInfo?.car) {
                                      isRightCar = linked_entities.some((entity) => {
                                          return entity.entity_type === 'car' && entity.object_id === signalInfo?.car;
                                      });
                                  }
                                  let isRightTrace = true;
                                  if (signalInfo?.trace) {
                                      isRightTrace = linked_entities.some((entity) => {
                                          return (
                                              entity.entity_type === 'trace' && entity.object_id === signalInfo?.trace
                                          );
                                      });
                                  }

                                  return isRightCar && isRightTrace;
                              }
                          })
                          ?.slice(0, 1)
                          ?.map((signal) => {
                              let { name } = signal;
                              let formattedSignal: FormattedSignal = Object.assign({ description: null }, signal);
                              formattedSignal.description = signals_descriptions?.[name] ?? null;

                              return formattedSignal;
                          })
                    : null;

                setFormattedSignal(formattedSignal ? formattedSignal[0] : null);
                setIsLoading(signalsResponse.meta === ABORT_ERROR_KEY);
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });

        if (signalInfo?.car) {
            request.current
                .exec(REQUESTS.GET_CARS, {
                    queryParams: { car_id: signalInfo?.car },
                })
                .then((carInfo) => {
                    let { cars, models } = carInfo;
                    let car = cars?.[0] ?? {};

                    car.model = models?.[car?.model_id]?.name || car?.model_id;
                    car.number = car.number?.toUpperCase();
                    car.taxi_park = car?.taxi_company?.name;

                    setCar(car);
                });
        }
    };

    let time = formattedSignal?.since ? shortDateTime(formattedSignal?.since) : EMPTY_DATA;

    let trace = formattedSignal?.linked_entities.find((entity) => entity.entity_type === 'trace') ?? null;
    let traceLink = '';
    if (trace) {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
        searchParams.set(MODAL_OBJECT_ID_CGI, trace.object_id);
        searchParams.set('back', `${location.pathname}?${location.search}`);

        traceLink = `${location.pathname}?${searchParams}`;
    }

    const signalDisplayName = formattedSignal?.description?.display_name as any;

    return (
        <>
            <GlobalSidebarHeader>
                <div className={style.signal_info}>
                    <Header2>
                        {isLoading ? (
                            <div className={style.signal_title_shimmer} />
                        ) : formattedSignal?.description?.display_name ? (
                            TRANSLATES[signalDisplayName] || signalDisplayName
                        ) : (
                            formattedSignal?.name ?? EMPTY_DATA
                        )}

                        {formattedSignal?.is_actual ? <div className={style.is_actual_icon} /> : null}
                    </Header2>
                    {isLoading ? (
                        <div className={style.signal_since_shimmer} />
                    ) : (
                        <div className={style.since}>{time}</div>
                    )}
                </div>
            </GlobalSidebarHeader>
            {error ? (
                <div className={style.error_container}>
                    <ErrorLabel simple />
                </div>
            ) : null}
            {!isLoading && !error && formattedSignal && car ? (
                <SignalEntityBlock
                    title={i18n('Car')}
                    link={`/cars/${car.id}`}
                >
                    <div className={style.car_info}>
                        <div className={style.car_name}>{car?.model ?? EMPTY_DATA}</div>
                        <div className={style.car_indicators}>
                            <CarNumberLabel carInfo={car ?? null} />
                            <Status statusType={car.status} />
                        </div>
                    </div>
                </SignalEntityBlock>
            ) : null}
            {!isLoading && !error && formattedSignal && trace ? (
                <SignalEntityBlock
                    title={i18n('Ride')}
                    link={traceLink}
                >
                    <div className={style.session_info}>
                        <SessionTime
                            oneRow
                            start={trace?.details?.since}
                            finish={trace?.details?.until}
                        />
                    </div>
                </SignalEntityBlock>
            ) : null}
        </>
    );
};

export default ModalSignalView;
