import * as React from 'react';
import { MutableRefObject, useRef } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { useRequestHandler } from 'hooks/useRequestHandler';

import { ICarImei, ICarInfo } from 'utils/car/types';

import { Spin } from 'shared/ui/Spin/Spin';

import { CARS_REQUESTS, REQUESTS } from 'components/Cars/request';
import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import { FormStep } from 'components/GlobalSidebar/ModalUpsertCarView/FormStep';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import { Input } from 'components/ui/Input';
import Header2 from 'components/ui/Text/Header2';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalUpsertCarView/index.i18n';

import styles from 'components/GlobalSidebar/ModalUpsertCarView/index.css';

interface IModalEditCarViewProps {
    carId: string | null;
}

export const ModalUpsertCarView = ({ carId }: IModalEditCarViewProps) => {
    let [modalType, setModalType] = React.useState<'create' | 'update'>('create');
    let textKeys = {
        update: {
            button: i18n('Save changes'),
            title: i18n('Edit car'),
        },

        create: {
            button: i18n('Create car'),
            title: i18n('Add car'),
        },
    };

    let request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: CARS_REQUESTS }));
    let requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_CARS,
            requestOptions: {
                queryParams: {
                    car_id: carId,
                    traits: ['ReportIMEI', 'ReportVIN', 'ReportShowAllModels'].join(','),
                },
            },
        };
    }, [carId]);
    let location = useLocation();
    let history = useHistory();

    let [isLoading, car, loadingError, getCar] = useRequestHandler<ICarInfo & ICarImei>(
        request?.current,
        requestOptions,
        (data) => {
            let car = data?.cars?.[0];

            setCarModel(data?.models[car?.model_id]?.name);
            setCarVin(car?.vin);
            setCarImei(car?.imei);
            setCarNumber(car?.number);

            return car;
        },
    );

    React.useEffect(() => {
        if (carId) {
            setModalType('update');
            getCar();
        }
    }, [carId]);

    let [carVin, setCarVin] = React.useState<string | undefined>();
    let [carModel, setCarModel] = React.useState<string | undefined>();
    let [carNumber, setCarNumber] = React.useState<string | undefined>();
    let [carImei, setCarImei] = React.useState<string | undefined>();

    let editRequestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.EDIT_CAR,
            requestOptions: {
                body: {
                    id: carId,
                    imei: carImei,
                    number: carNumber,
                },
            },
        };
    }, [carId, carVin, carModel, carImei, carNumber]);

    const onSuccessUpsert = () => {
        let urlSearchParams = new URLSearchParams(location?.search);
        urlSearchParams.delete(MODAL_OBJECT_TYPE_CGI);
        urlSearchParams.delete(MODAL_OBJECT_ID_CGI);

        history.push(`${location.pathname}?${urlSearchParams}`);

        return null;
    };

    let [isSubmitting, , submitError, editCar] = useRequestHandler<undefined>(
        request?.current,
        editRequestOptions,
        onSuccessUpsert,
    );

    const onSubmit = () => {
        editCar();
    };

    return (
        <div>
            <GlobalSidebarHeader>
                <Header2>{textKeys?.[modalType]?.title}</Header2>
            </GlobalSidebarHeader>
            <div className={styles.upsert_car_container}>
                {isLoading ? (
                    <div className={styles.centerSpin}>
                        <Spin />
                    </div>
                ) : (
                    <>
                        {loadingError ? (
                            <ErrorReloadLabel
                                title={i18n('Error while loading car')}
                                reloadFunction={getCar}
                            />
                        ) : (
                            <>
                                {submitError ? <ErrorReloadLabel title={i18n('Error while editing car')} /> : null}
                                <div className={styles.step_form}>
                                    <FormStep
                                        step={1}
                                        title={i18n('General information')}
                                        description={i18n('Car can be created after filling in these fields')}
                                    >
                                        <div className={styles.step}>
                                            <Input
                                                value={car?.vin}
                                                disabled
                                                onChange={(value) => {
                                                    setCarVin(value);
                                                }}
                                                label={i18n('VIN')}
                                            />

                                            <Input
                                                disabled
                                                value={carModel}
                                                onChange={(value) => {
                                                    setCarModel(value);
                                                }}
                                                label={i18n('Model')}
                                            />
                                        </div>
                                    </FormStep>
                                </div>
                                <div className={styles.step_form}>
                                    <FormStep
                                        step={2}
                                        title={i18n('Additional information')}
                                        description={i18n(
                                            'Car has already been created, filling in these fields is required to show all data on the car',
                                        )}
                                    >
                                        <div className={`${styles.step} ${styles.step2}`}>
                                            <div>
                                                <span className={styles.title}>{i18n('Basic information')}</span>
                                                <Input
                                                    value={carNumber}
                                                    onChange={(value) => {
                                                        setCarNumber(value);
                                                    }}
                                                    label={i18n('License plate number')}
                                                />
                                            </div>
                                            <div>
                                                <span className={styles.title}>{i18n('Telematics data')}</span>
                                                <Input
                                                    value={carImei}
                                                    onChange={(value) => {
                                                        setCarImei(value);
                                                    }}
                                                    label={i18n('IMEI')}
                                                />
                                            </div>
                                        </div>
                                    </FormStep>
                                </div>
                                <div className={styles.save_button_container}>
                                    <ControlButton
                                        fullWidth
                                        disabled={isLoading || !!loadingError}
                                        isLoading={isSubmitting}
                                        title={textKeys?.[modalType]?.button}
                                        onClick={onSubmit}
                                    />
                                </div>
                            </>
                        )}
                    </>
                )}
            </div>
        </div>
    );
};
