import * as React from 'react';
import { useEffect, useState } from 'react';

import { Input } from 'components/ui/Input';
import { IUser } from 'components/Users/types';

import { i18n } from 'components/GlobalSidebar/ModalUserView/UserForm/index.i18n';

import style from 'components/GlobalSidebar/ModalUserView/UserForm/index.css';

export const PHONE_START_SYMBOL = '+';

interface IUserFormProps {
    user?: IUser | null;
    errors?: {
        lastName?: string | null;
        firstName?: string | null;
        pName?: string | null;
        phone?: string | null;
        email?: string | null;
    };

    onChange?: (data: { lastName: string; firstName: string; pName: string; phone: string; email: string }) => void;
    disabled?: boolean;
}

const UserForm = ({ user, onChange, disabled, errors: errorsProps }: IUserFormProps) => {
    const [lastName, setLastName] = useState<string>('');
    const [firstName, setFirstName] = useState<string>('');
    const [pName, setPName] = useState<string>('');
    const [phone, setPhone] = useState<string>(PHONE_START_SYMBOL);
    const [email, setEmail] = useState<string>('');

    const [errors, setErrors] = useState<{
        lastName?: string | null;
        firstName?: string | null;
        pName?: string | null;
        phone?: string | null;
        email?: string | null;
    }>({});

    useEffect(() => {
        if (user) {
            let { last_name, first_name, pn, phone_number, email } = user;

            setLastName(last_name);
            setFirstName(first_name);
            setPName(pn);
            setPhone(phone_number);
            setEmail(email);
        }
    }, [user]);

    useEffect(() => {
        let newErrors = Object.assign({}, errors);
        ['lastName', 'firstName', 'pName', 'phone', 'email'].forEach((key) => {
            newErrors[key] = errorsProps?.[key] ?? errors?.[key] ?? null;
        });
        setErrors(newErrors);
    }, [errorsProps?.lastName, errorsProps?.firstName, errorsProps?.pName, errorsProps?.phone, errorsProps?.email]);

    const onFormChange = (type: string, value: string) => {
        let newErrors = Object.assign({}, errors);

        switch (type) {
            case 'lastName':
                setLastName(value);
                break;
            case 'firstName':
                setFirstName(value);
                break;
            case 'pName':
                setPName(value);
                break;
            case 'phone':
                setPhone(
                    value
                        ? value[0] === PHONE_START_SYMBOL
                            ? value
                            : `${PHONE_START_SYMBOL}${value}`
                        : !user || (user && user?.phone_number) //if EXISTING user has no number, it's possible to leave it empty
                        ? PHONE_START_SYMBOL
                        : '',
                );

                break;
            case 'email':
                setEmail(value);
                break;
        }

        setErrors(newErrors);
    };

    useEffect(() => {
        onChange?.({
            lastName,
            firstName,
            pName,
            phone,
            email,
        });
    }, [lastName, firstName, pName, phone, email]);

    return (
        <>
            <span className={style.form_title}>{i18n('Personal data')}</span>
            <div className={style.form}>
                <Input
                    value={lastName}
                    onChange={onFormChange.bind(null, 'lastName')}
                    required
                    disabled={disabled}
                    error={errors?.lastName ?? ''}
                    label={i18n('Surname')}
                />

                <Input
                    value={firstName}
                    onChange={onFormChange.bind(null, 'firstName')}
                    required
                    disabled={disabled}
                    error={errors?.firstName ?? ''}
                    label={i18n('Name')}
                />

                <Input
                    value={pName}
                    onChange={onFormChange.bind(null, 'pName')}
                    disabled={disabled}
                    error={errors?.pName ?? ''}
                    label={i18n('Middle name')}
                />

                <Input
                    value={phone}
                    onChange={onFormChange.bind(null, 'phone')}
                    required
                    disabled={disabled}
                    error={errors?.phone ?? ''}
                    label={i18n('Phone')}
                />

                <Input
                    value={email}
                    onChange={onFormChange.bind(null, 'email')}
                    disabled={disabled}
                    error={errors?.email ?? ''}
                    label={i18n('E-mail')}
                />
            </div>
        </>
    );
};

export default UserForm;
