import * as React from 'react';
import { MutableRefObject, useContext, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isShowUsersRides } from 'utils/isShowUsersRides';
import getFinish from 'utils/sessions/getFinish';
import { ISession } from 'utils/sessions/types';
import { showUserScoringInformer } from 'utils/showUserScoringInformer';

import { getUserRoles, UserStatusInfo } from 'entities/User';
import { UserRole } from 'entities/User/consts/UserRole';
import { UserStatus } from 'entities/User/consts/UserStatus';

import { GlobalSidebarHeader } from 'components/GlobalSidebar/GlobalSidebarHeader';
import BlockUserConfirmModal from 'components/GlobalSidebar/ModalUserView/BlockUserConfirmModal';
import DeleteUserConfirmModal from 'components/GlobalSidebar/ModalUserView/DeleteUserConfirmModal';
import UserForm, { PHONE_START_SYMBOL } from 'components/GlobalSidebar/ModalUserView/UserForm';
import formatPhoneValue from 'components/GlobalSidebar/ModalUserView/UserForm/formatPhoneValue';
import { CharType } from 'components/Informer';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { REQUESTS as REQUESTS_RIDES, RIDES_REQUESTS } from 'components/Rides/request';
import { RidesTable } from 'components/Rides/RidesTable';
import CommandButton from 'components/ui/Buttons/CommandButton';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import Tabs from 'components/ui/Tabs';
import { TabsSize } from 'components/ui/Tabs/types';
import Header2 from 'components/ui/Text/Header2';
import Widget from 'components/ui/Widget';
import { REQUESTS, USERS_REQUESTS } from 'components/Users/request';
import { IUser } from 'components/Users/types';
import { UserScoringInformer } from 'components/Users/UserScoringInformer/UserScoringInformer';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/GlobalSidebar/ModalUserView/index.i18n';

import coreStyle from 'components/Content/index.css';
import style from 'components/GlobalSidebar/ModalUserView/index.css';

const RIDES_TABLE_COLUMNS = ['date_time', 'mileage', 'duration', 'violations'];

const ModalUserView = ({ userId }: { userId: string | null }) => {
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [user, setUser] = useState<IUser | null>(null);
    const [isDriver, setIsDriver] = useState<boolean>(false);

    const [activeTab, setActiveTab] = useState<'rides' | 'settings' | null>('settings');

    const [isPhoneNumberExist, setIsPhoneNumberExist] = useState<boolean>(false);

    const [isDeleteConfirmOpen, setIsDeleteConfirmOpen] = useState<boolean>(false);
    const [isDeleting, setIsDeleting] = useState<boolean>(false);
    const [errorDeleting, setErrorDeleting] = useState<Error | null>(null);

    const [isBlockConfirmOpen, setIsBlockConfirmOpen] = useState<boolean>(false);
    const [isBlocking, setIsBlocking] = useState<boolean>(false);
    const [errorBlocking, setErrorBlocking] = useState<Error | null>(null);

    const [isEditing, setIsEditing] = useState<boolean>(false);

    const [userData, setUserData] = useState<{
        lastName: string;
        firstName: string;
        pName: string;
        phone: string;
        email: string;
    }>({
        lastName: '',
        firstName: '',
        pName: '',
        phone: '',
        email: '',
    });

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: USERS_REQUESTS }));
    let { current: requestRides }: React.MutableRefObject<RequestHelper> = React.useRef(
        new RequestHelper({ requestConfigs: RIDES_REQUESTS }),
    );

    let { notificationDispatch } = useContext(NotificationCenterContext) || {};

    let location = useLocation();
    let history = useHistory();

    useEffect(() => {
        return () => request.current.abort();
    }, []);

    useEffect(() => {
        getUser();
    }, [userId]);

    const getUser = () => {
        setIsLoading(true);
        setError(null);
        request.current.abort();

        request.current
            .exec(REQUESTS.GET_USERS, { body: { users_ids: userId } })
            .finally(() => {
                setIsLoading(false);
            })
            .then((response) => {
                let user: IUser = response?.members?.[0] ?? null;
                if (isShowUsersRides()) {
                    const isDriver = user?.roles?.includes('driver') ?? false;
                    setIsDriver(isDriver);
                    setActiveTab(isDriver ? 'rides' : 'settings');
                }
                setUser(user);
            })
            .catch(setError);
    };

    const openDeleteConfirm = () => {
        setIsDeleteConfirmOpen(true);
    };

    const closeDeleteConfirm = () => {
        setIsDeleteConfirmOpen(false);
    };

    const openBlockConfirm = () => {
        setIsBlockConfirmOpen(true);
    };

    const closeBlockConfirm = () => {
        setIsBlockConfirmOpen(false);
    };

    const onDeleteUser = () => {
        setIsDeleting(true);
        setErrorDeleting(null);

        request.current
            .exec(REQUESTS.DELETE_USER, { queryParams: { user_id: userId } })
            .finally(() => {
                setIsDeleting(false);
            })
            .then(() => {
                closeDeleteConfirm();
                let searchParams = new URLSearchParams(location.search);
                searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                searchParams.delete(MODAL_OBJECT_ID_CGI);
                history.push(`${location.pathname}?${searchParams}`);

                let notificationOptions = {
                    title: i18n('User is deleted'),
                };

                notificationDispatch(addNotification(notificationOptions));
            })
            .catch(setErrorDeleting);
    };

    const onBlockUser = (unblocking?: boolean) => {
        setIsBlocking(true);
        setErrorBlocking(null);

        let data = {
            id: user?.id,
            status: unblocking ? 'active' : 'blocked',
            last_name: user?.last_name,
            first_name: user?.first_name,
            pn: user?.pn,
            phone: user?.phone_number,
            email: user?.email,
        };

        request.current
            .exec(REQUESTS.EDIT_USER, { body: data })
            .then(() => {
                setIsBlocking(false);
                closeBlockConfirm();

                let searchParams = new URLSearchParams(location.search);
                searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                searchParams.delete(MODAL_OBJECT_ID_CGI);
                history.push(`${location.pathname}?${searchParams}`);

                let notificationOptions = {
                    title: unblocking ? i18n('User is unblocked') : i18n('User is blocked'),
                };

                notificationDispatch(addNotification(notificationOptions));
            })
            .catch((error) => {
                setIsBlocking(false);
                setErrorBlocking(error);
            });
    };

    const onSaveClick = () => {
        setIsEditing(true);

        let data = {
            id: user?.id,
            status: user?.status,
            last_name: userData.lastName,
            first_name: userData.firstName,
            pn: userData.pName,
            phone: userData.phone,
            email: userData.email,
        };

        isPhoneNumberValid()
            .then((isNumberValid) => {
                if (!isNumberValid) {
                    setIsPhoneNumberExist(true);
                    setIsEditing(false);
                } else {
                    request.current
                        .exec(REQUESTS.EDIT_USER, { body: data })
                        .then(() => {
                            setIsEditing(false);

                            let searchParams = new URLSearchParams(location.search);
                            searchParams.delete(MODAL_OBJECT_TYPE_CGI);
                            searchParams.delete(MODAL_OBJECT_ID_CGI);
                            history.push(`${location.pathname}?${searchParams}`);

                            let notificationOptions = {
                                title: i18n('User changed successfully'),
                            };

                            notificationDispatch(addNotification(notificationOptions));
                        })
                        .catch(() => {
                            setIsEditing(false);

                            let notificationOptions = {
                                title: i18n('Failed to change user'),
                                iconType: NotificationIconType.ERROR,
                            };

                            notificationDispatch(addNotification(notificationOptions));
                        });
                }
            })
            .catch(() => {
                setIsEditing(false);

                let notificationOptions = {
                    title: i18n('Failed to change user'),
                    iconType: NotificationIconType.ERROR,
                };

                notificationDispatch(addNotification(notificationOptions));
            });
    };

    const isPhoneNumberValid = () => {
        return new Promise((resolve, reject) => {
            if (user?.phone_number === userData.phone) {
                //If number isn't changed it's valid
                resolve(true);
            } else {
                request.current
                    .exec(REQUESTS.SEARCH, { queryParams: { has_all_of: formatPhoneValue(userData.phone) } })
                    .then((response) => {
                        let isUserExist = response?.objects?.users.filter((user) => user.status !== 'deleted')?.length;
                        resolve(!isUserExist);
                    })
                    .catch(reject);
            }
        });
    };

    const onUserFormDataChange = (userFormData) => {
        if (userFormData.phone !== userData.phone) {
            setIsPhoneNumberExist(false);
        }
        setUserData(userFormData);
    };

    const onTabChange = (link: 'rides' | 'settings') => {
        setActiveTab(link);
    };

    const getUserRides = async (since: number | null, until: number | null) => {
        return requestRides
            .exec(REQUESTS_RIDES.GET_RIDES, {
                queryParams: {
                    since,
                    until,
                    numdoc: 3,
                    user_id: userId ?? null,
                },
            })
            .then((response: { has_more: boolean; sessions: ISession[]; models: any }) => {
                let lastSession = response.sessions?.[response.sessions?.length - 1];

                return {
                    models: response.models,
                    sessions: response.sessions,
                    canGetMore: response.has_more,
                    currentUntil: response.has_more ? getFinish(lastSession)?.timestamp ?? null : null,
                };
            })
            .catch((error) => {
                throw error;
            });
    };

    let displayName = [user?.last_name, user?.first_name, user?.pn].filter((name) => name).join(' ');

    const isUserDataChanged =
        user?.last_name !== userData.lastName ||
        user?.first_name !== userData.firstName ||
        user?.pn !== userData.pName ||
        user?.phone_number !== userData.phone ||
        user?.email !== userData.email;

    let isRequiredFieldsEntered =
        userData.lastName && userData.firstName && userData.phone.replace(PHONE_START_SYMBOL, '');

    const isScoringVisible = showUserScoringInformer();

    let content: JSX.Element | null = null;
    switch (activeTab) {
        case 'rides':
            content = (
                <>
                    {isScoringVisible && (
                        <>
                            <UserScoringInformer
                                type={CharType.line}
                                userId={userId}
                            />

                            <h4 className={style.form_title}>{i18n('Last rides')}</h4>
                        </>
                    )}

                    <Widget
                        contentComponent={
                            <RidesTable
                                getRides={getUserRides}
                                columns={RIDES_TABLE_COLUMNS}
                            />
                        }
                    />
                </>
            );

            break;
        case 'settings':
            content = (
                <>
                    <div className={style.roles}>
                        <div className={style.title}>
                            <span className={style.main}>{i18n('User roles')}</span>
                            <span className={style.description}>
                                {i18n('Responsible for access to the admin panel and application')}
                            </span>
                        </div>
                        <div className={style.roles_list}>
                            {!isLoading ? (
                                user?.roles?.length ? (
                                    getUserRoles(user?.roles as UserRole[]).map((role) => (
                                        <div
                                            key={role}
                                            className={style.role_label}
                                        >
                                            {role}
                                        </div>
                                    ))
                                ) : (
                                    <span>{i18n('User has no roles')}</span>
                                )
                            ) : (
                                <>
                                    <div className={`${coreStyle.shimmer} ${style.role_label_shimmer}`} />
                                    <div className={`${coreStyle.shimmer} ${style.role_label_shimmer}`} />
                                </>
                            )}
                        </div>
                    </div>
                    <div className={style.separator} />
                    <UserForm
                        user={user}
                        onChange={onUserFormDataChange}
                        errors={{
                            phone: isPhoneNumberExist ? i18n('User with this phone number already exists') : '',
                        }}
                    />

                    {user?.status === 'active' || user?.status === 'blocked' ? (
                        <>
                            <div className={style.separator} />
                            <CommandButton
                                title={user?.status === 'blocked' ? i18n('Unblock user') : i18n('Block user')}
                                description={
                                    user?.status === 'blocked'
                                        ? i18n('The user will get access to the application')
                                        : i18n('The user will not be able to use the application')
                                }
                                onClick={openBlockConfirm}
                            />
                        </>
                    ) : null}
                    <div className={style.separator} />
                    <CommandButton
                        critical
                        title={i18n('Delete user')}
                        description={i18n('The user will lose all access forever')}
                        onClick={openDeleteConfirm}
                    />

                    {isUserDataChanged ? (
                        <div className={style.button_container}>
                            <ControlButton
                                fullWidth
                                disabled={!isRequiredFieldsEntered}
                                isLoading={isEditing}
                                title={i18n('Save changes')}
                                onClick={onSaveClick}
                            />
                        </div>
                    ) : null}
                </>
            );

            break;
    }

    return (
        <div>
            {error ? (
                <ErrorReloadLabel reloadFunction={getUser} />
            ) : (
                <>
                    <GlobalSidebarHeader>
                        <div className={style.header}>
                            {isLoading ? (
                                <>
                                    <div className={`${coreStyle.shimmer} ${style.name_shimmer}`} />
                                    <div className={`${coreStyle.shimmer} ${style.status_shimmer}`} />
                                </>
                            ) : (
                                <>
                                    <Header2
                                        className={style.name}
                                        title={displayName || EMPTY_DATA}
                                    >
                                        {displayName || EMPTY_DATA}
                                    </Header2>
                                    <UserStatusInfo status={user?.status as UserStatus} />
                                </>
                            )}
                        </div>
                    </GlobalSidebarHeader>
                    <div className={style.content_container}>
                        <table className={style.info_table}>
                            <tbody>
                                <tr>
                                    <td className={style.field}>{i18n('E-mail')}</td>
                                    <td
                                        className={style.value}
                                        title={user?.email || EMPTY_DATA}
                                    >
                                        {isLoading ? (
                                            <div className={`${coreStyle.shimmer} ${style.info_value_shimmer}`} />
                                        ) : (
                                            user?.email || EMPTY_DATA
                                        )}
                                    </td>
                                </tr>
                                <tr>
                                    <td className={style.field}>{i18n('Phone')}</td>
                                    <td
                                        className={style.value}
                                        title={user?.phone_number || EMPTY_DATA}
                                    >
                                        {isLoading ? (
                                            <div className={`${coreStyle.shimmer} ${style.info_value_shimmer}`} />
                                        ) : (
                                            user?.phone_number || EMPTY_DATA
                                        )}
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                        {isDriver && isShowUsersRides() ? (
                            <div className={style.tabs_container}>
                                <Tabs
                                    size={TabsSize.M}
                                    tight
                                    tabs={[
                                        { link: 'rides', name: i18n('Rides') },
                                        { link: 'settings', name: i18n('Settings') },
                                    ]}
                                    activeTab={activeTab}
                                    onChange={onTabChange}
                                />
                            </div>
                        ) : null}
                        {content}
                    </div>
                </>
            )}

            {isBlockConfirmOpen ? (
                <BlockUserConfirmModal
                    title={user?.status === 'blocked' ? i18n('Unblock user?') : i18n('Block user?')}
                    confirmButtonTitle={user?.status === 'blocked' ? i18n('Yes, unblock') : i18n('Yes, block')}
                    onConfirm={onBlockUser.bind(null, user?.status === 'blocked')}
                    onClose={closeBlockConfirm}
                    isLoading={isBlocking}
                    error={errorBlocking}
                />
            ) : null}
            {isDeleteConfirmOpen ? (
                <DeleteUserConfirmModal
                    onConfirm={onDeleteUser}
                    onClose={closeDeleteConfirm}
                    isLoading={isDeleting}
                    error={errorDeleting}
                />
            ) : null}
        </div>
    );
};

export default ModalUserView;
