import React, { useEffect, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_BACK, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_SESSION_TAB, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isCarsGroupsAccess } from 'utils/isCarsGroupsAccess';
import { isShowUsersSection } from 'utils/isShowUsersSection';

import ModalAddSignal from 'components/GlobalSidebar/ModalAddSignal';
import ModalAddUserView from 'components/GlobalSidebar/ModalAddUserView';
import ModalGroupView from 'components/GlobalSidebar/ModalGroupView';
import ModalSearchView from 'components/GlobalSidebar/ModalSearchView';
import ModalSessionView from 'components/GlobalSidebar/ModalSessionView';
import ModalSignalSettings from 'components/GlobalSidebar/ModalSignalSettings';
import ModalSignalView from 'components/GlobalSidebar/ModalSignalView';
import { ModalUpsertCarView } from 'components/GlobalSidebar/ModalUpsertCarView';
import ModalUserView from 'components/GlobalSidebar/ModalUserView';
import { ModalObjectTypes } from 'components/types';
import OverlayContainer, { OverlayIndex } from 'components/ui/OverlayContainer';

import CloseIcon from 'components/ui/Close/close.component.svg';

import style from 'components/GlobalSidebar/index.css';

const GlobalSidebar = () => {
    const [isShown, setIsShown] = useState<boolean>(false);
    const [isShowCloseButton, setIsShowCloseButton] = useState<boolean>(false);
    const [modalType, setModalType] = useState<ModalObjectTypes | null>(null);
    const [modalId, setModalId] = useState<string | null>(null);
    const [contentComponent, setContentComponent] = useState<any>(null);

    let location = useLocation();
    let history = useHistory();

    useEffect(() => {
        switch (modalType) {
            case ModalObjectTypes.SESSION:
                setContentComponent(<ModalSessionView sessionId={modalId} />);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.SIGNAL:
                setContentComponent(<ModalSignalView signalInfo={modalId} />);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.USER:
                setContentComponent(isShowUsersSection() ? <ModalUserView userId={modalId} /> : null);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.SEARCH:
                setContentComponent(<ModalSearchView />);
                setIsShowCloseButton(false);
                break;
            case ModalObjectTypes.UPSERT_CAR:
                setContentComponent(<ModalUpsertCarView carId={modalId} />);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.CAR_GROUP:
                setContentComponent(isCarsGroupsAccess() ? <ModalGroupView /> : null);
                setIsShowCloseButton(isCarsGroupsAccess());
                break;
            case ModalObjectTypes.ADD_SIGNAL:
                setContentComponent(<ModalAddSignal />);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.SIGNAL_SETTINGS:
                setContentComponent(<ModalSignalSettings />);
                setIsShowCloseButton(true);
                break;
            case ModalObjectTypes.ADD_USER:
                setContentComponent(isShowUsersSection() ? <ModalAddUserView /> : null);
                setIsShowCloseButton(true);
                break;
            default:
                setContentComponent(null);
                setIsShowCloseButton(false);
                break;
        }
    }, [modalType]);

    useEffect(() => {
        let urlSearchParams = new URLSearchParams(location?.search);
        let modalTypeNew = urlSearchParams.get(MODAL_OBJECT_TYPE_CGI);
        let modalIdNew = urlSearchParams.get(MODAL_OBJECT_ID_CGI);

        if (modalTypeNew !== modalType || modalIdNew !== modalIdNew) {
            if (modalTypeNew) {
                setIsShown(true);
                setModalType(modalTypeNew as ModalObjectTypes);
                setModalId(modalIdNew);
            } else {
                setIsShown(false);
                setModalType(null);
                setModalId(null);
            }
        }
    }, [location]);

    useEffect(() => {
        if (isShown) {
            document.body.style.overflow = 'hidden';
        } else {
            document.body.style.overflow = 'auto';
        }
    }, [isShown]);

    const onCloseClick = () => {
        let urlSearchParams = new URLSearchParams(location?.search);
        urlSearchParams.delete(MODAL_OBJECT_TYPE_CGI);
        urlSearchParams.delete(MODAL_OBJECT_ID_CGI);
        urlSearchParams.delete(MODAL_OBJECT_SESSION_TAB);
        urlSearchParams.delete(MODAL_BACK);

        history.push(`${location.pathname}?${urlSearchParams}`);
    };

    return (
        <>
            {isShown ? (
                <OverlayContainer
                    onClose={onCloseClick}
                    zIndex={OverlayIndex.sidebar}
                />
            ) : null}
            <div className={`${style.global_sidebar} ${!isShown ? style.hidden : ''}`}>
                {isShowCloseButton ? (
                    <div
                        className={`${style.close_button}`}
                        onClick={onCloseClick}
                    >
                        <CloseIcon />
                    </div>
                ) : null}
                {contentComponent}
            </div>
        </>
    );
};

export default GlobalSidebar;
