import React, { MutableRefObject, useRef } from 'react';
import { Chart, registerables } from 'chart.js';
import cn from 'classnames';

import { EMPTY_DATA } from 'constants/constants';

import { Spin } from 'shared/ui/Spin/Spin';

import DescriptionIcon from 'components/ui/DescriptionIcon';

import { i18n } from 'components/Informer/index.i18n';

import Icon from 'components/Informer/nodata.component.svg';

import styles from 'components/Informer/index.css';

interface IInfoItem {
    value: number | string;
}

interface IInfoItemComponentProps extends IInfoItem {
    isEmpty: boolean;
}

export enum CharType {
    line = 'line',
    bar = 'bar',
}

export interface IInformerProps {
    className?: string;
    data: any[];
    title: string;
    value: any;
    type: CharType;
    isLoading: boolean;
    hideHeader?: boolean;
    description?: string;
    options?: {
        hideLineGradient: boolean;
        yAxes?: any;
        borderColor?: any;
        tension?: number;
        customPadding?: boolean;
    };

    additionalDatasets?: { data: any[] }[];
    additionalHeaders?: any[];
}

export default function Informer(props: IInformerProps) {
    let {
        className,
        data,
        title,
        type,
        value,
        isLoading,
        hideHeader,
        options,
        description,
        additionalDatasets,
        additionalHeaders,
    } = props;
    let ref = React.useRef<HTMLCanvasElement>(null);
    React.useLayoutEffect(() => {
        Chart.register(...registerables);
        Chart.defaults.plugins.legend.display = false;
        Chart.defaults.plugins.tooltip.position = 'nearest';
        Chart.defaults.plugins.tooltip.displayColors = false;
    });

    React.useEffect(() => {
        data.length && ref?.current && initChart(ref?.current);
    }, [data, additionalDatasets?.[0]?.data?.length]);

    const chart: MutableRefObject<Chart | null> = useRef(null);

    const initChart = (ctx) => {
        const gradient_const = 250;
        let isLine = type === CharType.line;
        let gradient;
        if (isLine && !options?.hideLineGradient) {
            let ctx2: any = document.createElement('canvas').getContext('2d');
            gradient = ctx2?.createLinearGradient(0, 0, 0, gradient_const);
            gradient.addColorStop(0, 'rgba(0,0,0,0.2)');
            gradient.addColorStop(1, 'rgba(255,255,255,0.001)');
        }

        const defaultTension = 0.05;
        chart.current?.destroy();
        chart.current = new Chart(ctx, {
            type,
            data: {
                datasets: [
                    {
                        data,
                        label: title,
                        fill: false,
                        borderColor: options?.borderColor ?? 'rgb(18,21,21)',
                        borderWidth: 2,
                        animation: false,
                        borderRadius: 3,
                        tension: options?.tension || defaultTension,
                        backgroundColor: options?.borderColor ?? '#343434',
                        ...(isLine && !options?.hideLineGradient
                            ? {
                                  fill: {
                                      target: 'origin',
                                      above: gradient,
                                  },
                              }
                            : {}),
                    },

                    ...((additionalDatasets && additionalDatasets) || []),
                ],
            },

            options: {
                responsive: true,
                interaction: {
                    mode: 'index',
                    intersect: false,
                },

                scales: {
                    x: {
                        grid: {
                            color: 'white',
                        },
                    },

                    yAxes: {
                        ...(options?.yAxes
                            ? options?.yAxes
                            : {
                                  ticks: {
                                      count: 5,
                                      callback: function (value) {
                                          const kilo = 1000;
                                          const fraction = 2;
                                          value = Math.floor(+value);
                                          if (value > kilo) {
                                              return (value / kilo).toFixed(fraction) + 'k';
                                          }

                                          return value.toLocaleString('ru');
                                      },
                                  },
                              }),
                    },
                },

                plugins: {
                    tooltip: {
                        callbacks: {
                            label: (context) => {
                                const label = context.dataset.label || '';
                                const value = context.formattedValue || '';

                                return `${label}: ${value}`;
                            },
                        },
                    },
                },
            },
        });
    };
    let isEmpty = !data.length;
    let isCustomPadding = options?.customPadding;

    return (
        <div className={cn(styles.component, [isCustomPadding && styles.custom, className])}>
            {!hideHeader && (
                <div className={styles.header}>
                    <div className={styles.title}>
                        {title}
                        {description ? <DescriptionIcon description={description} /> : null}
                    </div>
                    <div className={styles.additional_top}>{additionalHeaders?.[0]}</div>
                    <div className={styles.value}>
                        <InfoItem
                            value={value}
                            isEmpty={isEmpty}
                        />
                    </div>
                    <div className={styles.additional_bottom}>{additionalHeaders?.[1]}</div>
                </div>
            )}

            <div className={styles.chart}>
                <div className={styles.center}>{isLoading ? <Spin /> : isEmpty ? <NoData /> : null}</div>
                {isEmpty ? null : (
                    <canvas
                        className={styles.canvas}
                        ref={ref}
                    />
                )}
            </div>
        </div>
    );
}

const InfoItem = ({ value, isEmpty }: IInfoItemComponentProps) => {
    return (
        <div className={styles.info_item}>
            <div className={styles.value}>{isEmpty ? EMPTY_DATA : value}</div>
        </div>
    );
};

const NoData = () => {
    return (
        <div className={styles.no_data}>
            <div className={styles.no_data_icon}>
                <Icon />
            </div>
            <div className={styles.no_data_label}>{i18n('There is no data')}</div>
        </div>
    );
};

export const InformersGrid = ({ children }) => {
    return <div className={styles.informers_grid}>{children}</div>;
};

export const DashedLabel = () => {
    return (
        <div className={styles.dashed_label}>
            {i18n('Average')}
            <div className={styles.dash} />
        </div>
    );
};
