import React from 'react';

import createGlobalAreas from 'utils/map/area/createGlobalAreas';
import createOtherAreas from 'utils/map/area/createOtherAreas';
import showArea from 'utils/map/area/showArea';
import { IShowItemsFilter } from 'utils/map/area/types';
import initMap from 'utils/map/map/initMap';

import { buildBoundControl } from 'components/Map/controls/bound';
import { buildZoomControl } from 'components/Map/controls/zoom';
import { IBaseMapProps } from 'components/Map/types';
import { IArea } from 'components/Zones/types';

import styles from 'components/Map/index.css';

export class BaseMap<P extends IBaseMapProps> extends React.Component<P> {
    map: ymaps.Map;
    defaultId: string = 'default_map_id';
    areas: ymaps.GeoObjectCollection;

    get id() {
        return this.props.options?.id || this.defaultId;
    }

    customInit(map: ymaps.Map) {
        this.map = map;
        if (!this.props.options.hideZoomControls) {
            buildZoomControl(this.map);
        }
        if (!this.props.options.hideBoundControls) {
            buildBoundControl(this.map);
        }
        this.areas = new ymaps.GeoObjectCollection();
        this.map.geoObjects.add(this.areas);

        this.props.disableDrag && this.map.behaviors.disable('drag');
    }

    componentDidUpdate(prevProps: Readonly<P>) {
        if (this.props.disableDrag !== prevProps.disableDrag) {
            if (this.props.disableDrag) {
                this.map.behaviors.disable('drag');
            } else {
                this.map.behaviors.enable('drag');
            }
        }
    }

    drawAreas(props?: {
        propertiesFunc?: (area: IArea) => any;
        optionsFunc?: (area: IArea) => any;
        createdAreaCallback?: (area: IArea, polygon: any) => any;
    }) {
        let { propertiesFunc, optionsFunc, createdAreaCallback } = props ?? {};
        this.areas?.removeAll();

        let extractArea = this.props.areas?.find((area) => area.area_tags.includes('extract'));

        createGlobalAreas({
            inner: extractArea,
            collection: this.areas,
            propertiesFunc,
            optionsFunc,
            createdAreaCallback,
        });

        createOtherAreas({
            areas: this.props.areas,
            collection: this.areas,
            extractArea,
            propertiesFunc,
            optionsFunc,
            createdAreaCallback,
        });
    }

    showAreas(filterItems: IShowItemsFilter) {
        showArea({
            ...filterItems,
            collection: this.areas,
        });
    }

    componentDidMount() {
        initMap({
            ...Object.assign({}, this.props.options, { id: this.id }),
            callback: (map: ymaps.Map) => {
                this.customInit(map);
            },
        });
    }

    componentWillUnmount() {
        this.map?.destroy();
    }

    render() {
        return (
            <div
                className={`${styles.map}`}
                id={this.id}
            />
        );
    }
}
