import { withRouter } from 'react-router-dom';
import isEqual from 'lodash.isequal';

import createCluster from 'utils/map/cluster/createCluster';
import getCgiMapOptions from 'utils/map/map/getCgiMapOptions';
import createPlacemark from 'utils/map/placemark/createPlacemark';
import getGPSCoordinates from 'utils/map/placemark/getGPSCoordinates';

import { BaseMap } from 'components/Map/BaseMap';
import { IBeacon } from 'components/Map/controls/BeaconPinLayout/types';
import { IMapProps } from 'components/Map/types';
import { Areas } from 'components/ui/Buttons/AreasButton/AreasWidget';

const SMOOTH_MAP_UPDATE = 500;
const CAR_ZOOM_MARGIN = 250;
const GPS_BEACON_ZOOM_MARGIN = 70;

class CarsMap extends BaseMap<IMapProps> {
    clusterer: ymaps.Clusterer;
    defaultId = 'cars_map_id';

    customInit(props) {
        super.customInit(props);
        this.clusterer = createCluster();
        this.map.geoObjects.add(this.clusterer);

        this.drawObjects(this.props.cars, this.props.beacon);
        // this.drawAreas();
        this.applyPolyFilters();
    }

    componentDidUpdate(prevProps: Readonly<IMapProps>) {
        super.componentDidUpdate(prevProps);

        if (!isEqual(this.props.cars, prevProps.cars) || !isEqual(this.props.beacon, prevProps.beacon)) {
            this.drawObjects(
                this.props.cars,
                this.props.beacon,
                prevProps.beacon === undefined,
                prevProps.beacon && this.props.beacon === undefined,
            );
        }

        if (!isEqual(this.props.areas, prevProps.areas)) {
            // this.drawAreas();
        }
        if (!isEqual(this.props.showAreas, prevProps.showAreas)) {
            this.applyPolyFilters();
        }
    }

    applyPolyFilters() {
        this.showAreas({
            [Areas.FORBIDDEN_TO_RIDE]: this.props.showAreas?.[Areas.FORBIDDEN_TO_RIDE],
            [Areas.FORBIDDEN_TO_PARK]: this.props.showAreas?.[Areas.FORBIDDEN_TO_PARK],
        });
    }

    drawObjects(cars, beacon: IBeacon | undefined, isInitialBeacon?: boolean, isBeaconRemoved?: boolean) {
        this.clusterer?.removeAll?.();
        if (cars && cars.length) {
            cars = cars.filter((car) => car.location);
            this.clusterer?.add(cars.map((car) => createPlacemark({ car, props: this.props })));
        }

        if (beacon?.location) {
            this.clusterer?.add(createPlacemark({ beacon, props: this.props }));
        }

        if (!beacon || isInitialBeacon) {
            if (this.props.options?.zoom && this.props.options?.zoom > 0 && isBeaconRemoved) {
                this.map?.setZoom(this.props.options.zoom);
            }
            this.setBounds(cars, beacon);
        }
    }

    setBounds(cars, beacon) {
        if (this.props?.alwaysCarCenter && !beacon) {
            // for centering single car on minimap
            this.map?.setCenter(getGPSCoordinates(cars[0]), this.map.getZoom(), {
                duration: SMOOTH_MAP_UPDATE,
            });

            return;
        }

        let { zoom, lon, lat } = getCgiMapOptions(this.props);

        if (!zoom && !lon && !lat) {
            let bounds = this.clusterer?.getBounds();
            let zoomMargin = beacon && this.props?.alwaysCarCenter ? GPS_BEACON_ZOOM_MARGIN : CAR_ZOOM_MARGIN;
            let options: any = {
                preciseZoom: true,
                zoomMargin,
                duration: SMOOTH_MAP_UPDATE,
            };

            bounds?.[0]?.[0] && this.map.setBounds(bounds, options);
        }
    }
}

export default withRouter(CarsMap);
