import isEqual from 'lodash.isequal';

import { onTimelineClick } from 'utils/map/trace/click';
import createPin from 'utils/map/trace/createPin';
import { createTrack2 } from 'utils/map/trace/createTrack';
import { onHover, onMouseEnterLeave } from 'utils/map/trace/hover';
import { IconTraceType, ITrackMapProps } from 'utils/map/trace/types';

import { BaseMap } from 'components/Map/BaseMap';

class TrackMap extends BaseMap<ITrackMapProps> {
    defaultId = 'track_map_id';
    track: ymaps.GeoObjectCollection;

    customInit(props) {
        super.customInit(props);

        this.track = new ymaps.GeoObjectCollection();
        this.map.geoObjects.add(this.track);

        this.track.events.add('hover', onHover);
        this.track.events.add(['mouseenter', 'mouseleave'], onMouseEnterLeave.bind(this, this.props.hover));

        this.track.events.add('click', onTimelineClick);

        this.buildTrack();
    }

    componentWillUnmount() {
        super.componentWillUnmount();
        this.track.events.remove('hover', onHover);
        this.track.events.remove(['mouseenter', 'mouseleave'], onMouseEnterLeave.bind(this, this.props.hover));
        this.track.events.remove('click', onTimelineClick);
    }

    componentDidUpdate(prevProps: Readonly<ITrackMapProps>) {
        if (!isEqual(this.props.timeline, prevProps.timeline)) {
            this.buildTrack();
        }
        if (!isEqual(this.props.hovered, prevProps.hovered)) {
            this.hover();
        }
        if (!isEqual(this.props.clicked, prevProps.clicked)) {
            this.click();
        }
    }

    hover() {
        this.track.events.fire('hover', { originId: this.props.hovered });
    }

    click() {
        this.track.events.fire('click', { originId: this.props.clicked });
    }

    buildTrack() {
        let {
            props: { timeline, session },
        } = this;

        this.track?.removeAll();
        if (timeline.length && this.track) {
            if (session?.track) {
                session.track?.features?.forEach?.((track) => {
                    this.track.add(createTrack2(track?.geometry?.coordinates, track.properties.violation));
                });
            }

            timeline.forEach((tml) => {
                if (tml.type === IconTraceType.SPEEDING) {
                    let pinProperties = {
                        center: tml.coords as number[],
                        type: IconTraceType.SPEEDING,
                        visible: false,
                        options: {
                            value: tml.meta?.violation?.translated || '',
                            is_hard: tml.meta?.violation?.is_hard,
                        },
                    };
                    this.track.add(createPin(pinProperties));
                } else {
                    this.track.add(
                        createPin({
                            center: tml.coords as number[],
                            type: tml.type,
                        }),
                    );
                }
            });

            let trackBound = this.track.getBounds();

            let options: any = { preciseZoom: true, zoomMargin: 15 };
            trackBound?.[0]?.[0] && this.map.setBounds(trackBound, options);
        }
    }
}

export default TrackMap;
