import isEqual from 'lodash.isequal';

import { IZoneEditorProps } from 'utils/map/trace/types';

import { BaseMap } from 'components/Map/BaseMap';

class ZoneEditor extends BaseMap<IZoneEditorProps> {
    geoObjectsCollection: ymaps.GeoObjectCollection;
    defaultId = 'zone_editor';
    mainColor = '#CF1722';

    customInit(props) {
        super.customInit(props);

        this.areas.events.add('boundschange', () => {
            this.setBounds();
        });

        this.drawAreas({
            propertiesFunc: (area) => {
                return { hintContent: area.area_title };
            },
            createdAreaCallback: (area, polygon) => {
                polygon.events.add('click', () => {
                    this.props.onZoneClick(area.area_title);
                });
            },
        });
        this.geoObjectsCollection = new ymaps.GeoObjectCollection(
            {},
            {
                editorDrawingCursor: 'crosshair',
                fillColor: this.mainColor,
                fillOpacity: 0.1,
                strokeColor: this.mainColor,
                strokeOpacity: 0.5,
                strokeWidth: 2,
            },
        );
        this.map.geoObjects.add(this.geoObjectsCollection);
        window.addEventListener('keyup', this.keyUp.bind(this));
    }

    componentDidUpdate(prevProps: Readonly<IZoneEditorProps>) {
        super.componentDidUpdate(prevProps);

        if (this.props.isCreateZoneMode !== prevProps.isCreateZoneMode) {
            if (this.props.isCreateZoneMode) {
                // @ts-ignore
                window.ymaps.ready(() => {
                    this.startDrawing();
                });
            } else {
                this.cancelEditing(true);
                this.geoObjectsCollection.removeAll();
            }
        }

        if (!isEqual(this.props.areas, prevProps.areas)) {
            this.drawAreas({
                propertiesFunc: (area) => {
                    return { hintContent: area.area_title };
                },
                createdAreaCallback: (area, polygon) => {
                    polygon.events.add('click', () => {
                        this.props.onZoneClick(area.area_title);
                    });
                },
            });
        }
    }

    componentWillUnmount() {
        super.componentWillUnmount();
        window.removeEventListener('keyup', this.keyUp.bind(this));
    }

    cancelEditing(finalCancel?: boolean) {
        let allAreasCoordinates: any[] = [];
        this.geoObjectsCollection?.each((poly: any) => {
            allAreasCoordinates.push(...poly?.geometry?.getCoordinates());
            poly.editor.stopDrawing();
            poly.editor.stopEditing();
        });

        if (!finalCancel) {
            this.props.onFinishCreatingCoordinates(allAreasCoordinates);
            this.startDrawing();
        }
    }

    keyUp(e: KeyboardEvent) {
        if (e.key === 'Escape' || e.key === 'Enter') {
            this.cancelEditing();
        }
    }

    startDrawing() {
        let newPolygon = new ymaps.Polygon([]);
        this.geoObjectsCollection.add(newPolygon);

        newPolygon.editor.options.set('menuManager', ([], {}) => {});

        newPolygon.editor.startDrawing();
    }

    setBounds() {
        // @ts-ignore
        window?.ymaps?.ready(() => {
            let bounds = this.areas?.getBounds();

            const ZOOM_MARGIN = 100;
            const DURATION = 300;

            let options: any = {
                preciseZoom: true,
                zoomMargin: ZOOM_MARGIN,
                duration: DURATION,
            };

            bounds?.[0]?.[0] && this.map.setBounds(bounds, options);
        });
    }
}

export default ZoneEditor;
