import React, { MutableRefObject, useRef } from 'react';
import { Link } from 'react-router-dom';

import { INotifyComponentItemProps, NotificationIconType } from 'components/NotificationCenter/types';
import CloseIcon from 'components/ui/Close';
import MainText3 from 'components/ui/Text/MainText3';
import Subhead1 from 'components/ui/Text/Subhead1';

import DoneIcon from 'components/NotificationCenter/NotifyItem/done.component.svg';
import ErrorIcon from 'components/NotificationCenter/NotifyItem/error.component.svg';

import styles from 'components/NotificationCenter/NotifyItem/index.css';

const delayToHide = 2000;
const delayToRemove = 300;

const icons = {
    [NotificationIconType.DONE]: DoneIcon,
    [NotificationIconType.ERROR]: ErrorIcon,
};

export default function NotifyItem(props: INotifyComponentItemProps) {
    let {
        onClose,
        title,
        link,
        iconType = NotificationIconType.DONE,
        button,
        delay = delayToHide,
        isPermanent,
        description,
    } = props;
    let [markAsDeleted, mark] = React.useState(false);
    let Icon = icons[iconType];
    const timer: MutableRefObject<any> = useRef(null);

    React.useEffect(() => {
        if (!isPermanent) {
            timer.current = setTimeout(() => {
                mark(true);
                timer.current = setTimeout(() => {
                    onClose();
                }, delayToRemove);
            }, delay);
        }

        return () => {
            clearTimeout(timer.current);
        };
    }, []);

    const onMouseEnter = () => {
        clearTimeout(timer.current);
    };

    const onMouseLeave = () => {
        if (!isPermanent) {
            timer.current = setTimeout(() => {
                mark(true);
                timer.current = setTimeout(() => {
                    onClose();
                }, delayToRemove);
            }, delay);
        }
    };

    return (
        <div
            className={`${styles.notify_item_variables} ${styles.notify_item} ${
                markAsDeleted ? styles.mark_as_deleted : ''
            }`}
            onMouseEnter={onMouseEnter}
            onMouseLeave={onMouseLeave}
        >
            <Icon className={styles.icon} />
            <div className={styles.content}>
                {title ? <Subhead1 className={styles.title}>{title}</Subhead1> : null}
                {description ? <MainText3 className={styles.description}>{description}</MainText3> : null}
                {link?.to && (
                    <MainText3>
                        <Link to={link.to}>{link.label}</Link>
                    </MainText3>
                )}
                {button ? (
                    <MainText3>
                        <span
                            className={styles.control}
                            onClick={button?.onClick}
                        >
                            {button?.label}
                        </span>
                    </MainText3>
                ) : null}
            </div>
            <CloseIcon
                className={styles.close}
                onClick={onClose}
            />
        </div>
    );
}
