import React, { useEffect, useState } from 'react';

import { EMPTY_DATA } from 'constants/constants';

import { IPortfolioData, KEY_LINK_MAP, SH_NORMAL_VALUE } from 'components/Portfolio/PortfolioTable/types';
import ITableValueType from 'components/types';
import DescriptionIcon from 'components/ui/DescriptionIcon';
import Table from 'components/ui/Table';
import { getNumberDisplayString } from 'components/ui/Table/getNumberDisplayString';
import ShiftLabel from 'components/ui/Table/ShiftLabel';
import TableLinkLabel from 'components/ui/Table/TableLinkLabel';
import { IHeaderInfo, ITableRowData } from 'components/ui/Table/types';

import { i18n } from 'components/Portfolio/PortfolioTable/index.i18n';

import style from 'components/Portfolio/index.css';
import styles from 'components/Portfolio/PortfolioTable/index.css';

interface IPortfolioTableProps {
    isLoading: boolean;
    error: null | Error;
    portfolioData: IPortfolioData | null;
    getPortfolio: () => void;
}

const PortfolioTable = ({ isLoading, error, portfolioData, getPortfolio }: IPortfolioTableProps) => {
    const [header, setHeader] = useState<IHeaderInfo[]>([]);

    const translates = React.useMemo(
        () => ({
            Портфель: i18n('Portfolio'),
            Показатель: i18n('Indicator'),
            'Все машины': i18n('All cars'),
            'Машины с телематикой': i18n('Cars with telematics'),
            Сдаваемость: i18n('Utilization'),
            'Supply Hours': i18n('Supply Hours'),
            Машины: i18n('Cars'),
            'GMV на 1 машину': i18n('GMV for 1 car'),
            'Комиссия на 1 машину': i18n('Commission for 1 car'),
            'Не выполняют SH': i18n('Do not perform SH'),
            'Имеют дату выдачи менее 60 дней': i18n('Have a date of issue less than 60 days'),
        }),

        [],
    );

    useEffect(() => {
        let carsCountTelematics = getTelematicsCarsCount(portfolioData);

        const PORTFOLIO_COMMON_HEADER = [
            { key: 'mark', displayName: i18n('Indicator') },
            { key: 'all_cars', displayName: i18n('All cars') },
        ];

        let header =
            carsCountTelematics > 0
                ? [
                      ...PORTFOLIO_COMMON_HEADER,
                      {
                          key: 'telematics_cars',
                          displayName: i18n('Cars with telematics'),
                      },
                  ]
                : PORTFOLIO_COMMON_HEADER;

        setHeader(header);
    }, [portfolioData]);

    const getTelematicsCarsCount = (portfolioData) => {
        let carsCountData = portfolioData?.portfolio?.find?.((item) => item?.name === 'cars_count');

        return carsCountData?.telematics_cars?.value ?? null;
    };

    const buildTableRows = (): ITableRowData[] => {
        let values = portfolioData?.portfolio ?? [];

        return values.map((value, index) => {
            let { name, display_name, type, all_cars, telematics_cars } = value;

            let cellDisplayName = translates[display_name as any] || display_name;

            //Hard code of sh normal value
            if (name === 'no_sh') {
                cellDisplayName = `${cellDisplayName ?? ''} (${SH_NORMAL_VALUE})`;
            }

            return {
                data: {
                    mark: (
                        <div className={style.row_name}>
                            {cellDisplayName || name || EMPTY_DATA}
                            {getRowDescription(name) !== null ? (
                                <DescriptionIcon description={getRowDescription(name)} />
                            ) : null}
                        </div>
                    ),

                    all_cars: buildTableValueCell(all_cars, type, name, 'all_cars'),
                    telematics_cars: buildTableValueCell(telematics_cars, type, name, 'telematics_cars'),
                },

                meta: { key: name ?? index },
            };
        });
    };

    const getRowDescription = (key: string) => {
        switch (key) {
            case 'utilization':
                return i18n('Percentage of days during which cars appeared in the YT system');
            case 'sh':
                return i18n('Average time spent by a car in active statuses');
            case 'cars_count':
                return i18n('Number of cars issued by a specific leasing company');
            case 'park_commission':
                return i18n('Average amount of earnings of taxi companies on commission from trips');
            case 'no_sh':
                return i18n(
                    'The number of cars that have not worked in the service for a sufficient number of hours over the period',
                );
            case 'gmv':
                return i18n('Average value of all successful orders');
            default:
                return null;
        }
    };

    const buildTableValueCell = (
        value: { value: any; old_value: any } | undefined,
        type: ITableValueType,
        name: string,
        category: 'all_cars' | 'telematics_cars',
    ) => {
        let diff = value && 'old_value' in value ? getRowDiff(name, type, value?.value, value?.old_value) : null;
        let cellContent = value ? (
            <div className={style.portfolio_cell}>
                <span className={style.current_value}>{getNumberDisplayString(value?.value, type)}</span>
                {diff ? (
                    <ShiftLabel
                        shift_type={diff.type}
                        value={diff.diff}
                        isRiseIsNegative={diff.isRiseIsNegative}
                    />
                ) : null}
            </div>
        ) : null;

        return cellContent ? (
            KEY_LINK_MAP?.[name]?.[category] ? (
                <TableLinkLabel link={KEY_LINK_MAP[name][category]}>{cellContent}</TableLinkLabel>
            ) : (
                cellContent
            )
        ) : (
            EMPTY_DATA
        );
    };

    const getRowDiff = (name: string, type: ITableValueType, value: any, oldValue: any) => {
        const FULL_PERCENTS = 100;
        const getAbsoluteDiff = () => value - oldValue;
        const getPercentageDiff = () => {
            let absoluteDiff = getAbsoluteDiff();

            return oldValue !== 0 ? (absoluteDiff * FULL_PERCENTS) / oldValue : null;
        };
        const isRiseIsNegative = name === 'no_sh';

        switch (name) {
            case 'park_commission':
            case 'cars_count':
            case 'gmv':
                return {
                    type: ITableValueType.PERCENT,
                    diff: getPercentageDiff(),
                };

            default:
                return { type, diff: getAbsoluteDiff(), isRiseIsNegative };
        }
    };

    return (
        <>
            <h3 className={styles.title}>{i18n('Portfolio')}</h3>

            <Table
                isLoading={isLoading}
                error={error}
                header={header}
                tableData={buildTableRows()}
                reloadFunction={getPortfolio}
            />
        </>
    );
};

export default PortfolioTable;
