import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { EMPTY_DATA } from 'constants/constants';

import { PortfolioComparePeriod } from 'entities/Portfolio/consts/PortfolioComparePeriod';
import { getPortfolioComparePeriod } from 'entities/Portfolio/helpers/getPortfolioComparePeriod/getPortfolioComparePeriod';

import { PORTFOLIO_REQUESTS, REQUESTS } from 'components/Portfolio/request';
import { Direction } from 'components/ui/DescriptionIcon';
import Table from 'components/ui/Table';
import { getShiftNumberDisplayString } from 'components/ui/Table/getShiftNumberDisplayString';
import TableLinkLabel from 'components/ui/Table/TableLinkLabel';
import { IHeaderInfo, ITableRowData } from 'components/ui/Table/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Portfolio/TaxiParksTable/index.i18n';

import style from 'components/Portfolio/index.css';
import shiftLabelStyle from 'components/ui/Table/ShiftLabel/index.css';

export const TAXI_PARK_MAX_NAME_LENGTH = 20;
const PAGE_SIZE = 10;

interface ITaxiParksTableProps {
    comparePeriod: PortfolioComparePeriod; //if all - period is last 60 days
}

const TaxiParksTable = ({ comparePeriod }: ITaxiParksTableProps) => {
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const [taxiParksData, setTaxiParksData] = useState<any[] | null>(null);
    const [header, setHeader] = useState<IHeaderInfo[]>([]);
    const [canGetMore, setCanGetMore] = useState<boolean>(false);
    const [isMoreLoading, setIsMoreLoading] = useState<boolean>(false);
    const [page, setPage] = useState<number>(1);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: PORTFOLIO_REQUESTS }));

    const cityTranslates = React.useMemo(() => {
        return {
            Москва: i18n('Moscow'),
            Питер: i18n('St. Petersburg'),
        };
    }, []);

    const TAXI_PARKS_HEADER: IHeaderInfo[] = [
        { key: 'taxi_park', displayName: i18n('Taxi park') },
        { key: 'city', displayName: i18n('City') },
        { key: 'score', displayName: i18n('Scoring') },
        { key: 'utilization', displayName: i18n('Utilization') },
        { key: 'cars_count', displayName: i18n('Cars') },
        {
            key: 'sh',
            displayName: i18n('SH'),
            description: i18n('Average time spent by a car in active statuses'),
            descriptionDirection: Direction.LEFT,
        },

        {
            key: 'park_commission',
            displayName: i18n('Commission'),
            description: i18n('The amount of earnings of taxi companies on commission from trips for 1 car'),
            descriptionDirection: Direction.LEFT,
        },

        {
            key: 'gmv',
            displayName: i18n('GMV'),
            description: i18n(
                'The total cost of all successful orders, including subsidies, discounts, coupons, YT commission for 1 car',
            ),

            descriptionDirection: Direction.LEFT,
        },

        { key: 'row_type', displayName: '' },
    ];

    useEffect(() => {
        getTaxiParks(page);

        return () => request.current.abort();
    }, []);

    useEffect(() => {
        getTaxiParks(page);
    }, [comparePeriod]);

    const getTaxiParks = (page?: number, isMore = false) => {
        request.current.abort();

        if (isMore) {
            setIsMoreLoading(true);
        } else {
            setIsLoading(true);
        }
        setError(null);

        request.current
            .exec(REQUESTS.GET_TAXI_PARKS, {
                queryParams: { page_number: page, page_size: PAGE_SIZE },
                body: getPortfolioComparePeriod(comparePeriod),
            })
            .then((response) => {
                let {
                    parks: taxiParksDataResponse = [],
                    can_get_more_pages: canGetMore = false,
                    page_number: page = 0,
                } = response;

                let taxiParksDataNew = isMore
                    ? [...(taxiParksData || []), ...taxiParksDataResponse]
                    : taxiParksDataResponse;
                let isShowScoreColumn =
                    (taxiParksData?.some((taxiPark) => taxiPark?.all_cars?.hasOwnProperty('score')) &&
                        comparePeriod === null) ||
                    false;

                if (isMore) {
                    setIsMoreLoading(response.meta === ABORT_ERROR_KEY);
                } else {
                    setIsLoading(response.meta === ABORT_ERROR_KEY);
                }

                setTaxiParksData(taxiParksDataNew);
                setCanGetMore(canGetMore);
                setPage(page);
                setHeader(
                    isShowScoreColumn
                        ? TAXI_PARKS_HEADER
                        : TAXI_PARKS_HEADER.filter((header) => header.key !== 'score'),
                );
            })
            .catch((error) => {
                if (isMore) {
                    setIsMoreLoading(false);
                } else {
                    setIsLoading(false);
                }
                setError(error);
            });
    };

    const getParkTaxNumber = (parkId: string): string | null => {
        let parkIdSplit = (parkId ?? '').split('_');
        let parkIdLastPart = parkIdSplit[parkIdSplit.length - 1];

        return parkIdLastPart && !isNaN(+parkIdLastPart) ? parkIdLastPart : null;
    };

    const buildTableRows = (): ITableRowData[] => {
        return (
            taxiParksData?.map((taxiParkValue) => {
                let { name, park_id, city, all_cars, telematics_cars } = taxiParkValue;
                let { cars_count: allCarsCountObj } = all_cars ?? {};
                let { cars_count: telematicsCarsCountObj } = telematics_cars ?? {};

                let nameDisplay =
                    name?.length > TAXI_PARK_MAX_NAME_LENGTH
                        ? `${name?.slice(0, TAXI_PARK_MAX_NAME_LENGTH)}...`
                        : name ?? EMPTY_DATA;

                let parkTaxNumber = getParkTaxNumber(park_id ?? '');

                return {
                    data: {
                        taxi_park: (
                            <div
                                className={style.park_name}
                                title={name}
                            >
                                <TableLinkLabel link={`/cars?park_id=${park_id}`}>
                                    <div>
                                        <div className={style.name_display}>{nameDisplay}</div>
                                    </div>
                                </TableLinkLabel>
                                {parkTaxNumber ? (
                                    <div className={style.name_tax_number_display}>
                                        {i18n('TIN')}: {parkTaxNumber}
                                    </div>
                                ) : null}
                            </div>
                        ),

                        city: city ? cityTranslates[city] || city : EMPTY_DATA,
                        score: buildTableValueCell(taxiParkValue, 'score', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        utilization: buildTableValueCell(taxiParkValue, 'utilization', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        cars_count: buildTableValueCell(taxiParkValue, 'cars_count', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        sh: buildTableValueCell(taxiParkValue, 'sh', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        park_commission: buildTableValueCell(taxiParkValue, 'park_commission', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        gmv: buildTableValueCell(taxiParkValue, 'gmv', {
                            all_cars: allCarsCountObj,
                            telematics_cars: telematicsCarsCountObj,
                        }),

                        row_type: (
                            <div className={`${style.row_type} ${style.car_type_row_description}`}>
                                <div className={style.all_cars_value}>
                                    <div className={style.value}>{i18n('all cars')}</div>
                                </div>
                                {telematicsCarsCountObj.value > 0 ? (
                                    <div className={style.telematics_cars_value}>
                                        <div className={style.value}>{i18n('with telematics')}</div>
                                    </div>
                                ) : null}
                            </div>
                        ),
                    },

                    meta: {
                        key: `${name}_${parkTaxNumber}`,
                    },
                };
            }) ?? []
        );
    };

    const buildTableValueCell = (
        taxiParkValue: any,
        key: string,
        carsCount: Record<string, { value: any; old_value: any }>,
    ) => {
        let { all_cars, telematics_cars } = taxiParkValue;
        let { value: valueAll, old_value: valueOldAll, type: valueTypeAll } = all_cars?.[key] ?? {};
        let {
            value: valueTelematics,
            old_value: valueOldTelematics,
            type: valueTypeTelematics,
        } = telematics_cars?.[key] ?? {};
        let { all_cars: allCarsCount, telematics_cars: telematicsCarsCount } = carsCount;

        if (['park_commission', 'gmv'].includes(key)) {
            valueAll = allCarsCount?.value === 0 ? 0 : valueAll / allCarsCount?.value;
            valueOldAll =
                valueOldAll !== null
                    ? allCarsCount?.old_value === 0
                        ? 0
                        : valueOldAll / allCarsCount?.old_value
                    : null;

            valueTelematics = telematicsCarsCount?.value === 0 ? 0 : valueTelematics / telematicsCarsCount?.value;
            valueOldTelematics =
                valueOldTelematics !== null
                    ? telematicsCarsCount?.old_value === 0
                        ? 0
                        : valueOldTelematics / telematicsCarsCount?.old_value
                    : null;
        }

        let diffAll = getRowDiff(valueAll, valueOldAll);
        let diffTelematics = getRowDiff(valueTelematics, valueOldTelematics);

        return (
            <>
                <div className={style.all_cars_value}>
                    {valueAll !== null && valueAll !== undefined && diffAll !== 0 && diffAll !== null ? (
                        <div className={style.arrow_container}>
                            <div
                                className={`${shiftLabelStyle.arrow_icon} ${
                                    diffAll < 0
                                        ? shiftLabelStyle.arrow_icon_decrease
                                        : shiftLabelStyle.arrow_icon_increase
                                } ${shiftLabelStyle.big}`}
                            />
                        </div>
                    ) : null}
                    <span className={style.value}>{getShiftNumberDisplayString(valueAll, valueTypeAll)}</span>
                </div>
                {telematicsCarsCount?.value > 0 ? (
                    <div className={style.telematics_cars_value}>
                        {valueTelematics !== null &&
                        valueTelematics !== undefined &&
                        diffTelematics !== 0 &&
                        diffTelematics !== null ? (
                            <div className={style.arrow_container}>
                                <div
                                    className={`${shiftLabelStyle.arrow_icon} ${
                                        diffTelematics < 0
                                            ? shiftLabelStyle.arrow_icon_decrease
                                            : shiftLabelStyle.arrow_icon_increase
                                    }`}
                                />
                            </div>
                        ) : null}

                        <span className={style.value}>
                            {getShiftNumberDisplayString(valueTelematics, valueTypeTelematics)}
                        </span>
                    </div>
                ) : null}
            </>
        );
    };

    const getRowDiff = (value: any, oldValue: any) => {
        return oldValue !== null && oldValue !== undefined ? value - oldValue : null;
    };

    const onGetMoreClick = () => {
        getTaxiParks(page + 1, true);
    };

    return (
        <>
            <h3 className={`${style.taxi_park_title}`}>{i18n('All taxi parks')}</h3>

            <Table
                isLoading={isLoading}
                error={error}
                getMore={{
                    canGetMore,
                    onGetMoreClick,
                    isMoreLoading,
                }}
                header={header}
                tableData={buildTableRows() ?? []}
                reloadFunction={getTaxiParks.bind(null, 1)}
            />
        </>
    );
};

export default TaxiParksTable;
