import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { DEBOUNCE_TIMEOUT } from 'constants/constants';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { PortfolioFilters } from 'widgets/PortfolioFilters';

import { PortfolioHeaderTitle } from 'entities/Portfolio';
import { INITIAL_PORTFOLIO_FILTERS } from 'entities/Portfolio/consts/filters';
import { PortfolioComparePeriod } from 'entities/Portfolio/consts/PortfolioComparePeriod';
import { PortfolioFilter } from 'entities/Portfolio/consts/PortfolioFilter';
import { getPortfolioComparePeriod } from 'entities/Portfolio/helpers/getPortfolioComparePeriod/getPortfolioComparePeriod';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import PortfolioTable from 'components/Portfolio/PortfolioTable';
import { IPortfolioData } from 'components/Portfolio/PortfolioTable/types';
import { PORTFOLIO_REQUESTS, REQUESTS } from 'components/Portfolio/request';
import TaxiParksTable from 'components/Portfolio/TaxiParksTable';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../request-helper/src';

export interface PortfolioFiltersOptions {
    [PortfolioFilter.COMPARE_PERIOD]: Nullable<string>;
}

const Portfolio = () => {
    const [filters, setFilters] = useState<PortfolioFiltersOptions>(INITIAL_PORTFOLIO_FILTERS);
    const [portfolioData, setPortfolioData] = useState<IPortfolioData | null>(null);
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: PORTFOLIO_REQUESTS }));
    const debounceTimeout: MutableRefObject<any> = useRef(null);

    useEffect(() => {
        getPortfolio();

        return () => {
            request.current.abort();
            clearTimeout(debounceTimeout.current);
        };
    }, []);

    useEffect(() => {
        if (debounceTimeout.current !== null) {
            clearTimeout(debounceTimeout.current);
        }
        debounceTimeout.current = setTimeout(() => {
            getPortfolio();
        }, DEBOUNCE_TIMEOUT);
    }, [filters]);

    const getPortfolio = () => {
        setIsLoading(true);
        setError(null);
        request.current.abort();
        request.current
            .exec(REQUESTS.GET_PORTFOLIO, {
                body: getPortfolioComparePeriod(
                    filters[PortfolioFilter.COMPARE_PERIOD] as Nullable<PortfolioComparePeriod>,
                ),
            })
            .then((portfolioData) => {
                setPortfolioData(portfolioData);
                setIsLoading(portfolioData?.meta === ABORT_ERROR_KEY);
            })
            .catch((error) => {
                setError(error);
                setIsLoading(false);
            });
    };

    return (
        <SectionLayout
            header={
                <Header
                    title={<PortfolioHeaderTitle portfolioData={portfolioData} />}
                    withoutBorder
                />
            }
            filters={
                <PortfolioFilters
                    offsetTop={HEADER_HEIGHT}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ContentContainer withSidebar>
                <Widget contentContainer>
                    <PortfolioTable
                        getPortfolio={getPortfolio}
                        portfolioData={portfolioData}
                        isLoading={isLoading}
                        error={error}
                    />

                    <TaxiParksTable comparePeriod={filters[PortfolioFilter.COMPARE_PERIOD] as PortfolioComparePeriod} />
                </Widget>
            </ContentContainer>
        </SectionLayout>
    );
};

export default Portfolio;
