import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';

import { ONE_DAY, ONE_SECOND } from 'constants/constants';

import { ReportPeriod } from 'entities/Report/consts/ReportPeriod';
import { ReportSection } from 'entities/Report/consts/ReportSection';

import { getDateEndDay } from 'shared/helpers/getDateEndDay/getDateEndDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';

import ReportModal from 'components/Reports/ReportModal';
import { IReport } from 'components/Reports/types';
import Card from 'components/ui/Card';

import { i18n } from 'components/Reports/ReportCard/index.i18n';

import EditIcon from 'components/Reports/ReportCard/edit.component.svg';

export interface IReportCardProps {
    report: IReport;
    getReports: () => void;
}

const ReportCard = ({ report, getReports }: IReportCardProps) => {
    let { labels = [], section, reportName, reportDescription, filterValues, period } = report ?? {};
    let history = useHistory();

    const [isEditModalOpen, setIsEditModalOpen] = useState<boolean>(false);

    const onClick = () => {
        let searchParams = new URLSearchParams();

        if (period) {
            let currentDay = new Date();
            let since = new Date(getDateStartDay(currentDay).getTime() - period * ONE_DAY);
            let until = getDateEndDay(new Date(currentDay));

            searchParams.set('since', Math.trunc(+since / ONE_SECOND).toString());
            searchParams.set('until', Math.trunc(+until / ONE_SECOND).toString());
        }

        Object.entries(filterValues).forEach((entry: [string, string | number | boolean]) => {
            let [key, value] = entry;

            if (value !== null && value !== undefined) {
                searchParams.set(key, value.toString());
            }
        });

        history.push(`${section}?${searchParams}`);
    };

    const onEditModalClose = () => {
        setIsEditModalOpen(false);
        getReports();
    };

    let periodLabel: string[] = [];
    switch (period) {
        case ReportPeriod.DAY:
            periodLabel.push(i18n('Day'));
            break;
        case ReportPeriod.WEEK:
            periodLabel.push(i18n('Week'));
            break;
        case ReportPeriod.MONTH:
            periodLabel.push(i18n('Month'));
            break;
        case ReportPeriod.YEAR:
            periodLabel.push(i18n('Year'));
            break;
    }

    const onEditButtonClick = () => {
        setIsEditModalOpen(true);
    };

    let displaySection = '';
    switch (section) {
        case ReportSection.SIGNALS:
            displaySection = i18n('Signals');
            break;
        case ReportSection.RIDES:
            displaySection = i18n('Rides');
            break;
        case ReportSection.CARS:
            displaySection = i18n('Cars');
            break;
    }

    return (
        <>
            <Card
                onClick={onClick}
                name={reportName}
                icons={[{ icon: <EditIcon />, onClick: onEditButtonClick }]}
                labels={[...(labels ?? []), ...(displaySection ? [displaySection] : []), ...periodLabel]}
                description={reportDescription}
            />

            {isEditModalOpen ? (
                <ReportModal
                    editMode
                    onClose={onEditModalClose}
                    report={report}
                />
            ) : null}
        </>
    );
};

export default ReportCard;
