import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { ONE_SECOND } from 'constants/constants';

import { isValidJSONString } from 'utils/isValidJSONString';

import { ReportPeriod } from 'entities/Report/consts/ReportPeriod';
import { ReportSection } from 'entities/Report/consts/ReportSection';

import { Path } from 'shared/consts/Path';

import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { REQUESTS, SETTINGS_REQUESTS } from 'components/Reports/request';
import { IReport, REPORT_SETTINGS_FIELD } from 'components/Reports/types';
import ControlButton from 'components/ui/Buttons/ControlButton';
import ErrorLabel from 'components/ui/ErrorLabel';
import { Input } from 'components/ui/Input';
import { Modal } from 'components/ui/Modal';
import { RadioButton } from 'components/ui/RadioButton';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Reports/ReportModal/index.i18n';

import style from 'components/Reports/ReportModal/index.css';

interface IReportModalFormField {
    report_name: string;
    period: number | null;
    report_description: string;
}

interface IReportModalProps {
    onClose: () => void;
    report?: IReport;
    filterValues?: any;
    section?: ReportSection;
    editMode?: boolean;
}

const REPORTS_WITH_PERIOD = [ReportSection.SIGNALS];

const ReportModal = ({ onClose, filterValues, section, editMode, report: reportProps }: IReportModalProps) => {
    const [isLoading, setisLoading] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }));

    const isReportWithPeriod =
        (section && REPORTS_WITH_PERIOD.includes(section)) ||
        (reportProps?.section && REPORTS_WITH_PERIOD.includes(reportProps?.section));

    let { notificationDispatch } = React.useContext(NotificationCenterContext) || {};

    const [formValue, setFormValue] = useState<IReportModalFormField>({
        report_name: '',
        period: ReportPeriod.WEEK,
        report_description: '',
    });

    const PERIODS_OPTIONS = [
        { value: ReportPeriod.DAY, placeholder: i18n('Day') },
        { value: ReportPeriod.WEEK, placeholder: i18n('Week') },
        { value: ReportPeriod.MONTH, placeholder: i18n('Month') },
        { value: ReportPeriod.YEAR, placeholder: i18n('Year') },
    ];

    useEffect(() => {
        let formValueNew = {
            report_name: reportProps?.reportName ?? '',
            period: reportProps?.period ?? ReportPeriod.WEEK,
            report_description: reportProps?.reportDescription ?? '',
        };

        setFormValue(formValueNew);
    }, [reportProps]);

    const onFormChange = (type: string, value: string) => {
        let newFormValue = Object.assign({}, formValue);
        newFormValue[type] = value;

        setFormValue(newFormValue);
    };

    const getReports = () => {
        setisLoading(true);
        setError(null);

        return request.current
            .exec(REQUESTS.GET_SETTINGS, {
                queryParams: { fields: REPORT_SETTINGS_FIELD },
            })
            .then((settingsArray) => {
                let reportsSetting =
                    settingsArray?.settings?.find((settingsItem) => settingsItem.id === REPORT_SETTINGS_FIELD) ?? [];

                return isValidJSONString(reportsSetting.value) ? JSON.parse(reportsSetting.value) : [];
            })
            .catch((error) => {
                setError(error);
                setisLoading(false);
            });
    };

    const saveReports = (reports: IReport[]) => {
        setisLoading(true);
        setError(null);

        return request.current
            .exec(REQUESTS.SET_SETTINGS, {
                body: {
                    id: REPORT_SETTINGS_FIELD,
                    value: JSON.stringify(reports),
                },
            })
            .then((response) => {
                setisLoading(response?.meta === ABORT_ERROR_KEY);
                onClose();
            })
            .catch((error) => {
                setisLoading(false);
                setError(error);
            });
    };

    const onSaveClick = () => {
        getReports().then((reports) => {
            let { period, report_name, report_description } = formValue;

            let timestamp = Math.trunc(+new Date() / ONE_SECOND);
            let id = `${section}_${timestamp}`;

            let report: IReport = {
                id,
                section: section,
                reportName: report_name,
                reportDescription: report_description || '',
                timestamp,
                period: isReportWithPeriod ? period : null,
                filterValues: filterValues,
            };

            let newReports = [...reports, report];
            saveReports(newReports).then(() => {
                notificationDispatch(
                    addNotification({
                        title: i18n('Report was saved'),
                        link: {
                            label: i18n('Go to Reports section'),
                            to: Path.REPORTS,
                        },
                    }),
                );
            });
        });
    };

    const onRemoveClick = () => {
        getReports().then((reports) => {
            let newReports = (reports ?? []).filter((report) => report.id !== reportProps?.id);
            saveReports(newReports).then(() => {
                notificationDispatch(
                    addNotification({
                        title: i18n('Report was deleted'),
                    }),
                );
            });
        });
    };

    const onEditClick = () => {
        getReports().then((reports) => {
            let newReports = (reports ?? []).map((report) => {
                if (report.id === reportProps?.id) {
                    let { period, report_name, report_description } = formValue;

                    let report: IReport = {
                        id: reportProps?.id ?? '',
                        section: reportProps?.section,
                        reportName: report_name,
                        reportDescription: report_description || '',
                        timestamp: reportProps?.timestamp ?? 0,
                        period: isReportWithPeriod ? period : null,
                        filterValues: reportProps?.filterValues ?? {},
                    };

                    return report;
                } else {
                    return report;
                }
            });

            saveReports(newReports).then(() => {
                notificationDispatch(
                    addNotification({
                        title: i18n('Report was saved'),
                    }),
                );
            });
        });
    };

    return (
        <Modal
            onClose={onClose}
            title={editMode ? i18n('Edit report') : i18n('Save report')}
            contentComponent={
                <div>
                    <div className={style.inputs_container}>
                        <Input
                            value={formValue.report_name}
                            label={i18n('Report name')}
                            onChange={onFormChange.bind(null, 'report_name')}
                        />

                        {isReportWithPeriod ? (
                            <RadioButton
                                secondary
                                values={PERIODS_OPTIONS}
                                selectedValue={formValue.period}
                                onChange={onFormChange.bind(null, 'period')}
                            />
                        ) : null}
                        <Input
                            value={formValue.report_description}
                            label={i18n('Description')}
                            onChange={onFormChange.bind(null, 'report_description')}
                        />
                    </div>
                    <div className={style.separator} />
                    {error ? (
                        <div className={style.error_container}>
                            <ErrorLabel simple />
                        </div>
                    ) : null}
                    {editMode ? (
                        <>
                            <div className={style.buttons_container}>
                                <ControlButton
                                    fullWidth
                                    secondary
                                    critical
                                    disabled={!formValue.report_name || isLoading}
                                    title={i18n('Delete report')}
                                    onClick={onRemoveClick}
                                />
                            </div>
                            <div className={style.separator} />
                        </>
                    ) : null}
                    <div className={style.buttons_container}>
                        <ControlButton
                            fullWidth
                            disabled={!formValue.report_name || isLoading}
                            title={editMode ? i18n('Save changes') : i18n('Save')}
                            onClick={editMode ? onEditClick : onSaveClick}
                        />

                        {editMode ? (
                            <ControlButton
                                fullWidth
                                tertiary
                                disabled={isLoading}
                                title={i18n("Don't save changes")}
                                onClick={onClose}
                            />
                        ) : null}
                    </div>
                </div>
            }
        />
    );
};

export default ReportModal;
