import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import { isValidJSONString } from 'utils/isValidJSONString';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { ReportsFilters } from 'widgets/ReportsFilters';

import { INITIAL_REPORTS_FILTERS } from 'entities/Report/consts/filters';
import { ReportSection } from 'entities/Report/consts/ReportSection';
import { ReportsFilter } from 'entities/Report/consts/ReportsFilter';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import ReportsContent from 'components/Reports/ReportsContent';
import { REQUESTS, SETTINGS_REQUESTS } from 'components/Reports/request';
import { IReport, REPORT_SETTINGS_FIELD } from 'components/Reports/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Reports/index.i18n';

export interface ReportsFiltersOptions {
    [ReportsFilter.SECTION]: Nullable<string>;
    [ReportsFilter.PERIOD]: Nullable<string>;
}

const Reports = () => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }));
    const [reports, setReports] = useState<IReport[]>([]);
    const [filters, setFilters] = useState<ReportsFiltersOptions>(INITIAL_REPORTS_FILTERS);
    const [filteredReports, setFilteredReports] = useState<IReport[]>([]);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);

    useEffect(() => {
        getReports();

        return () => request.current.abort();
    }, []);

    useEffect(() => {
        getFilteredReports(reports, filters);
    }, [reports, filters]);

    const getReports = () => {
        setIsLoading(true);
        setError(null);
        request.current.abort();

        request.current
            .exec(REQUESTS.GET_SETTINGS, {
                queryParams: { fields: REPORT_SETTINGS_FIELD },
            })
            .then((settingsArray) => {
                let reportsSetting =
                    settingsArray?.settings?.find((settingsItem) => settingsItem.id === REPORT_SETTINGS_FIELD) ?? [];
                let reports = isValidJSONString(reportsSetting.value) ? JSON.parse(reportsSetting.value) : [];

                setReports(reports);
                setIsLoading(settingsArray?.meta === ABORT_ERROR_KEY);
            })
            .catch((error) => {
                setError(error);
                setIsLoading(false);
            });
    };

    const getFilteredReports = (
        reports: IReport[],
        { section: sectionFilter, period: periodFilter }: ReportsFiltersOptions,
    ) => {
        const filteredReports = reports.filter(({ section, period }) => {
            let isFilteredBySection =
                sectionFilter?.length && sectionFilter !== 'all'
                    ? sectionFilter.includes(section as ReportSection)
                    : true;
            let isFilteredByPeriod =
                period && periodFilter?.length && periodFilter !== 'all'
                    ? periodFilter.includes(period.toString())
                    : true;

            return isFilteredBySection && isFilteredByPeriod;
        });

        setFilteredReports(filteredReports);
    };

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Reports')}
                    withoutBorder
                />
            }
            filters={
                <ReportsFilters
                    offsetTop={HEADER_HEIGHT}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ContentContainer
                bodyScroll
                withSidebar
            >
                <ReportsContent
                    filteredReports={filteredReports}
                    reports={reports}
                    getReports={getReports}
                    error={error}
                    isLoading={isLoading}
                />
            </ContentContainer>
        </SectionLayout>
    );
};
export default Reports;
