import { EMPTY_DATA, VERY_DANGEROUS_SESSION_BORDER } from 'constants/constants';

import { isValidJSONString } from 'utils/isValidJSONString';
import getDuration from 'utils/sessions/getDuration';
import getFinish from 'utils/sessions/getFinish';
import getStart from 'utils/sessions/getStart';
import { ISession } from 'utils/sessions/types';
import { showCarDriverInRides } from 'utils/showCarDriverInRides';
import { getUserFullName } from 'utils/user/getUserFullName';

import { getRideBookingDetails } from 'entities/Ride/helpers/getRideBookingDetails/getRideBookingDetails';

import { getSessionDurationString } from 'components/Cars/CarCard/CarSessionsTable/getSessionDurationString';
import { getSessionMileageString } from 'components/Cars/CarCard/CarSessionsTable/getSessionMileageString';
import { TABLE_VIEW_SETTINGS_KEY_PREFIX } from 'components/Cars/CarsSidebar/CustomizeTableButton';
import { getRidesTableHeaders } from 'components/Rides/RidesTable/constants';
import { getSessionTime } from 'components/Signals/SignalsTable/getSessionTime';
import { REQUESTS } from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/request';

import { ABORT_ERROR_KEY } from '../../../../request-helper/src';

import { i18n } from 'components/Rides/RidesTable/downloadRidesTableXLSX.i18n';

const MAX_COUNT_REQUESTS = 15;

const buildTable = (rides: ISession[], models: any) => {
    return rides.map((session) => {
        let violations: string[] = [];

        if (session?.is_speeding) {
            violations.push(i18n('Speeding'));
        }

        if (session.aggressive_events) {
            session.aggressive_events.forEach((event) => {
                switch (event?.kind) {
                    case 'acceleration':
                        violations.push(i18n('Harsh acceleration'));
                        break;
                    case 'braking':
                        violations.push(i18n('Harsh braking'));
                        break;
                    case 'straight_lateral_acceleration':
                        violations.push(i18n('Harsh lane change'));
                        break;
                    case 'turning_lateral_acceleration':
                        violations.push(i18n('Harsh turn'));
                        break;
                    default:
                        violations.push(i18n('Dangerous maneuver'));
                        break;
                }
            });
        }

        const rideBookingDetails = getRideBookingDetails(session);
        let exportRideDetails = {};

        if (rideBookingDetails) {
            exportRideDetails = {
                ...rideBookingDetails,
                offer_options: rideBookingDetails.offer_options.join(', ') || EMPTY_DATA,
            };
        }

        return {
            date_time: getSessionTime(getStart(session)?.timestamp, getFinish(session)?.timestamp),
            driver: session.user_details ? getUserFullName(session.user_details) : EMPTY_DATA,
            car_model: models?.[session?.car?.model_id]?.name || session?.car?.model_id,
            car_number: session?.car?.number ?? session?.car?.vin,
            duration: getSessionDurationString(getDuration(session)),
            mileage: getSessionMileageString(session),
            score: session?.is_aggressive
                ? session?.aggressive_score !== undefined
                    ? session?.aggressive_score < VERY_DANGEROUS_SESSION_BORDER
                        ? i18n('Dangerous ride')
                        : i18n('Very dangerous ride')
                    : i18n('Dangerous ride')
                : EMPTY_DATA,
            violations: violations.join(', '),
            ...exportRideDetails,
        };
    });
};

export const downloadRidesTableXLSX = async (request, getData) => {
    let ridesTableHeaders = getRidesTableHeaders();
    try {
        let headerColumns = await request.exec(REQUESTS.GET_SETTINGS).then((settingsArray) => {
            let tableViewCarsSettings =
                settingsArray?.settings?.find(
                    (settingsItem) => settingsItem.id === `${TABLE_VIEW_SETTINGS_KEY_PREFIX}rides`,
                ) ?? [];
            let tableSettings = isValidJSONString(tableViewCarsSettings.value)
                ? JSON.parse(tableViewCarsSettings.value)?.[0]
                : null;

            // @todo: use filterRidesTableColumns
            if (tableSettings && !showCarDriverInRides()) {
                tableSettings.columns = tableSettings.columns.filter((key) => key !== 'driver');
            }

            return { columns: tableSettings?.columns ?? ridesTableHeaders.map(({ key }) => key) };
        });

        let rides: ISession[] = [];
        let models: any = [];
        let canGetMore = true;
        let currentUntil: number | undefined = undefined;
        let requestCount = 0;

        do {
            ++requestCount;
            let response = await getData(currentUntil).then((response) => {
                if (response.meta !== ABORT_ERROR_KEY) {
                    return response;
                }
            });
            if (response) {
                canGetMore = response.has_more;
                rides.push(...(response.sessions ?? []));
                models = {
                    ...models,
                    ...response.models,
                };

                let lastSession = response.sessions?.[response.sessions?.length - 1];
                currentUntil = response.has_more ? getFinish(lastSession)?.timestamp ?? undefined : undefined;
            }
        } while (canGetMore && requestCount < MAX_COUNT_REQUESTS);

        let headers = headerColumns?.columns?.length
            ? ridesTableHeaders.filter((carsHeaderItem) => headerColumns?.columns.includes(carsHeaderItem.key))
            : [];

        return { headers, table: buildTable(rides, models), name: `rides_${Date.now()}` };
    } catch (e) {
        throw e;
    }
};
