import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import {
    EMPTY_DATA,
    MODAL_BACK,
    MODAL_OBJECT_ID_CGI,
    MODAL_OBJECT_TYPE_CGI,
    ONE_SECOND,
    VERY_DANGEROUS_SESSION_BORDER,
} from 'constants/constants';

import { isUserProfilePageFlag } from 'utils/isUserProfilePageFlag';
import { isValidJSONString } from 'utils/isValidJSONString';
import getDuration from 'utils/sessions/getDuration';
import getFinish from 'utils/sessions/getFinish';
import getSessionId from 'utils/sessions/getSessionId';
import getStart from 'utils/sessions/getStart';
import { ISession } from 'utils/sessions/types';
import { showCarDriverInRides } from 'utils/showCarDriverInRides';
import { getUserFullName } from 'utils/user/getUserFullName';

import { getRideBookingDetails } from 'entities/Ride/helpers/getRideBookingDetails/getRideBookingDetails';

import { Path } from 'shared/consts/Path';

import { getSessionDurationString } from 'components/Cars/CarCard/CarSessionsTable/getSessionDurationString';
import { getSessionMileageString } from 'components/Cars/CarCard/CarSessionsTable/getSessionMileageString';
import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { CarNumberLabel } from 'components/Cars/CarNumberLabel';
import { TABLE_VIEW_SETTINGS_KEY_PREFIX } from 'components/Cars/CarsSidebar/CustomizeTableButton';
import { REQUESTS, SETTINGS_REQUESTS } from 'components/Reports/request';
import { RidesFiltersOptions } from 'components/Rides/index';
import { getRidesTableHeaders } from 'components/Rides/RidesTable/constants';
import { ModalObjectTypes } from 'components/types';
import { Label, LabelIndicatorTypes } from 'components/ui/Label';
import { ScoringLabel, ScoringLabelTypes } from 'components/ui/ScoringLabel';
import Table from 'components/ui/Table';
import TableLink from 'components/ui/Table/TableLink';
import { IHeaderInfo, ITableRowData } from 'components/ui/Table/types';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Rides/RidesTable/index.i18n';

import style from 'components/Rides/RidesTable/index.css';

interface IRidesTableProps {
    filters?: RidesFiltersOptions;
    columns?: string[];
    hasDriverColumn?: boolean;
    getRides: (
        since: number | null,
        until: number | null,
        userId?: string | null,
    ) => Promise<{ models: any; sessions: ISession[]; canGetMore: boolean; currentUntil: number | null }>;
}

const AGGRESSIVE_EVENTS: Record<string, string> = {
    acceleration: i18n('Harsh acceleration'),
    braking: i18n('Harsh braking'),
    straight_lateral_acceleration: i18n('Harsh lane change'),
    turning_lateral_acceleration: i18n('Harsh turn'),
};

const LABEL_ERROR = { type: LabelIndicatorTypes.ERROR };
const LABEL_OFFER_OPTION = false;

export const RidesTable = (props: IRidesTableProps) => {
    let { getRides, columns, hasDriverColumn = true, filters } = props;

    let [isLoading, setLoading] = React.useState<boolean>(true);
    let [error, setError] = React.useState<Error | null>(null);
    let [tableRows, setTableRows] = React.useState<ITableRowData[]>([]);
    let [headers, setHeaders] = React.useState<IHeaderInfo[]>([]);
    let [canGetMore, setCanGetMore] = React.useState<boolean>(false);
    let [currentUntil, setCurrentUntil] = React.useState<number | null>(null);
    let [isMoreLoading, setMoreLoading] = React.useState<boolean>(false);
    let history = useHistory();
    let location = useLocation();

    const request: React.MutableRefObject<RequestHelper> = React.useRef(
        new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }),
    );

    let onGetMoreClick = () => {
        setMoreLoading(true);
        let urlSearchParams = new URLSearchParams(location?.search);
        let userId = urlSearchParams.get('userId');
        let since = urlSearchParams.get('since') ? +(urlSearchParams.get('since') as string) : null;

        getRides(since, currentUntil, userId)
            .then((response) => {
                setTableRows([...tableRows, ...buildTableRows(response.sessions, response.models)]);
                setCanGetMore(response.canGetMore);
                setMoreLoading(false);
                setCurrentUntil(response.currentUntil);
            })
            .catch((error) => {
                setError(error);
            });
    };

    let getRidesForTable = () => {
        let urlSearchParams = new URLSearchParams(location?.search);
        let userId = urlSearchParams.get('userId');
        let since = urlSearchParams.get('since') ? +(urlSearchParams.get('since') as string) : null;
        let until = urlSearchParams.get('until')
            ? +(urlSearchParams.get('until') as string)
            : Math.floor(+Date.now() / ONE_SECOND);

        getRides(since, until, userId)
            .then((response) => {
                setTableRows(buildTableRows(response.sessions, response.models));

                return response;
            })
            .then((response) => {
                setCanGetMore(response.canGetMore);
                setCurrentUntil(response.currentUntil);
                setLoading(false);
            })
            .catch((error) => {
                setError(error);
                setLoading(false);
            });
    };

    let getHeaders = () => {
        let ridesTableHeaders = getRidesTableHeaders(hasDriverColumn);

        return request.current.exec(REQUESTS.GET_SETTINGS).then((settingsArray) => {
            let tableViewCarsSettings =
                settingsArray?.settings?.find(
                    (settingsItem) => settingsItem.id === `${TABLE_VIEW_SETTINGS_KEY_PREFIX}rides`,
                ) ?? [];
            let tableSettings = isValidJSONString(tableViewCarsSettings.value)
                ? JSON.parse(tableViewCarsSettings.value)?.[0]
                : null;

            if (!tableSettings?.columns?.length) {
                tableSettings = {
                    columns: ridesTableHeaders.map((header) => header.key),
                };
            }

            // @todo: use filterRidesTableColumns
            if (!showCarDriverInRides()) {
                tableSettings.columns = tableSettings.columns.filter((key) => key !== 'driver');
            }

            if (columns?.length) {
                tableSettings.columns = tableSettings.columns.filter((key) => columns?.includes(key));
            }

            setHeaders(ridesTableHeaders.filter((header) => tableSettings?.columns.includes(header.key)));
        });
    };

    React.useEffect(() => {
        setLoading(true);
        getHeaders().then(() => {
            getRidesForTable();
        });
    }, [filters?.since, filters?.until, columns]);

    let generateSearchParams = (params: { key; value }[]) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_BACK, `${location.pathname}?${searchParams}`);
        params.forEach(({ key, value }) => {
            searchParams.set(key, value);
        });

        return searchParams;
    };

    let buildTableRows = (sessions: ISession[], models: any): ITableRowData[] => {
        return sessions.map((session, index) => {
            let session_id = getSessionId(session);
            const userId = session.user_details?.id;
            let searchParams = generateSearchParams([
                {
                    key: MODAL_OBJECT_TYPE_CGI,
                    value: ModalObjectTypes.USER,
                },

                {
                    key: MODAL_OBJECT_ID_CGI,
                    value: userId,
                },
            ]);

            let user_link = isUserProfilePageFlag()
                ? `${Path.USERS}/${userId}/profile`
                : `${location.pathname}?${searchParams}`;
            let title = (session.user_details && getUserFullName(session.user_details)) || '';

            const ridesDetails = getRideBookingDetails(session);

            let tableRideDetails = {};

            if (ridesDetails) {
                tableRideDetails = {
                    ...ridesDetails,
                    offer_options: ridesDetails.offer_options.length ? (
                        <div className={style.label_container}>
                            {ridesDetails.offer_options.map((title, ind) => (
                                <Label
                                    key={ind}
                                    background
                                    indicator={LABEL_OFFER_OPTION}
                                    title={title}
                                />
                            ))}
                        </div>
                    ) : (
                        EMPTY_DATA
                    ),
                };
            }

            return {
                data: {
                    ...tableRideDetails,
                    date_time: (
                        <SessionTime
                            start={getStart(session)?.timestamp}
                            finish={getFinish(session)?.timestamp}
                        />
                    ),

                    driver:
                        title !== EMPTY_DATA ? (
                            <TableLink
                                key={userId}
                                link={user_link}
                            >
                                <Label
                                    background
                                    title={title}
                                    bgClassName={style.user}
                                />
                            </TableLink>
                        ) : (
                            EMPTY_DATA
                        ),

                    car_model: models?.[session?.car?.model_id]?.name || session?.car?.model_id,
                    car_number: (
                        <TableLink
                            key={session?.car?.id}
                            link={`/cars/${session?.car?.id}`}
                        >
                            <CarNumberLabel
                                carInfo={session.car}
                                onlyText
                                shortVIN
                            />
                        </TableLink>
                    ),

                    duration: getSessionDurationString(getDuration(session)),
                    mileage: getSessionMileageString(session),
                    score: session?.is_aggressive ? (
                        session?.aggressive_score !== undefined ? (
                            session?.aggressive_score < VERY_DANGEROUS_SESSION_BORDER ? (
                                <ScoringLabel
                                    type={ScoringLabelTypes.warning}
                                    title={i18n('Dangerous ride')}
                                />
                            ) : (
                                <ScoringLabel
                                    type={ScoringLabelTypes.negative}
                                    title={i18n('Very dangerous ride')}
                                />
                            )
                        ) : (
                            <ScoringLabel
                                type={ScoringLabelTypes.warning}
                                title={i18n('Dangerous ride')}
                            />
                        )
                    ) : null,
                    violations: (
                        <div className={style.label_container}>
                            {session?.is_speeding ? (
                                <Label
                                    background
                                    indicator={LABEL_ERROR}
                                    title={i18n('Speeding')}
                                />
                            ) : null}
                            {session.aggressive_events
                                ?.map((event) => event.kind)
                                ?.filter((value, index, _arr) => _arr.indexOf(value) == index)
                                ?.map((event_kind, key) => {
                                    const title = AGGRESSIVE_EVENTS[event_kind] || i18n('Dangerous maneuver');

                                    return (
                                        <Label
                                            key={key}
                                            background
                                            indicator={LABEL_ERROR}
                                            title={title}
                                        />
                                    );
                                })}
                        </div>
                    ),
                },

                meta: {
                    key: session_id ?? index.toString(),
                    className: style.car_session_row,
                    onClick: onRowClick.bind(null, session_id),
                },
            };
        });
    };

    let onRowClick = (sessionId: string) => {
        let searchParams = generateSearchParams([
            {
                key: MODAL_OBJECT_TYPE_CGI,
                value: ModalObjectTypes.SESSION,
            },

            {
                key: MODAL_OBJECT_ID_CGI,
                value: sessionId,
            },
        ]);

        history.push(`${location.pathname}?${searchParams}`);
    };

    return (
        <Table
            isLoading={isLoading}
            error={error}
            getMore={{
                canGetMore,
                onGetMoreClick,
                isMoreLoading,
            }}
            header={headers}
            tableData={tableRows}
            reloadFunction={() => {
                setLoading(true);
                getHeaders().then(() => {
                    getRidesForTable();
                });
            }}
            onSortChange={() => undefined}
        />
    );
};
