import * as React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';

import getFinish from 'utils/sessions/getFinish';
import { ISession } from 'utils/sessions/types';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { RidesContentHeader } from 'widgets/RidesContentHeader';
import { RidesFilters } from 'widgets/RidesFilters';

import { FilterDateOptions } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { INITIAL_RIDES_FILTERS } from 'entities/Ride/consts/filters';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import { REQUESTS, RIDES_REQUESTS } from 'components/Rides/request';
import { RidesTable } from 'components/Rides/RidesTable';

import { RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Rides/index.i18n';

export type RidesFiltersOptions = FilterDateOptions;

enum RideFilters {
    SINCE = 'since',
    UNTIL = 'until',
    NUMDOC = 'numdoc',
    USER_ID = 'userId',
}

const RIDES_NUMDOC = 50;

export type RideFiltersType = Record<RideFilters, string | number | boolean | null>;

const Rides = () => {
    let request: React.MutableRefObject<RequestHelper> = React.useRef(
        new RequestHelper({ requestConfigs: RIDES_REQUESTS }),
    );

    let [columns, setColumns] = React.useState<string[]>([]);
    let [filters, setFilters] = React.useState<RidesFiltersOptions>(INITIAL_RIDES_FILTERS);

    let fetchRides = (options: Partial<RideFiltersType>) => {
        return request.current
            .exec(REQUESTS.GET_RIDES, {
                queryParams: {
                    ...options,
                    user_id: options[RideFilters.USER_ID],
                    show_visible_sessions: options[RideFilters.USER_ID] ? null : true,
                },
            })
            .then((response) => {
                return response;
            })
            .catch((error) => {
                throw error;
            });
    };

    let getRides = async (since: number | null, until: number | null, userId: string | null) => {
        return fetchRides({
            since,
            until,
            numdoc: RIDES_NUMDOC,
            userId,
        })
            .then((response: { has_more: boolean; sessions: ISession[]; models: any }) => {
                let lastSession = response.sessions?.[response.sessions?.length - 1];

                return {
                    models: response.models,
                    sessions: response.sessions,
                    canGetMore: response.has_more,
                    currentUntil: response.has_more ? getFinish(lastSession)?.timestamp ?? null : null,
                };
            })
            .catch((error) => {
                throw error;
            });
    };

    const onTableColumnsChangeHandler = React.useCallback((columns: string[]) => {
        setColumns(columns);
    }, []);

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Rides')}
                    withoutBorder
                />
            }
            filters={
                <RidesFilters
                    offsetTop={HEADER_HEIGHT}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ContentContainer
                bodyScroll
                withSidebar
            >
                <Widget contentContainer>
                    <RidesContentHeader
                        filters={filters}
                        getData={fetchRides}
                        onCarsTableColumnsChange={onTableColumnsChangeHandler}
                    />

                    <RidesTable
                        filters={filters}
                        columns={columns}
                        getRides={getRides}
                    />
                </Widget>
            </ContentContainer>
        </SectionLayout>
    );
};

export default withRouter<RouteComponentProps, any>(Rides);
