import React, { MutableRefObject, useRef } from 'react';

import { isSwitchEnv } from 'utils/isSwitchEnv';
import { showSettingsTagsFlag } from 'utils/showSettingsTags';

import { EnvSwitcher } from 'features/EnvSwitcher';
import { UserOrganizationTags } from 'features/UserOrganizationTags/ui/UserOrganizationTags/UserOrganizationTags';

import { useCurrentUser } from 'shared/hooks/useCurrentUser/useCurrentUser';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { Spin } from 'shared/ui/Spin/Spin';
import { Widget } from 'shared/ui/Widget/Widget';

import { REQUESTS, SETTINGS_REQUESTS } from 'components/Settings/request';
import RoleItem from 'components/Settings/SettingsRoles/RoleItem';
import { IRoleItem } from 'components/Settings/SettingsRoles/types';
import Header3 from 'components/ui/Text/Header3';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Settings/SettingsRoles/index.i18n';

import styles from 'components/Settings/SettingsRoles/index.css';

export default function SettingsRoles() {
    let [isLoading, setLoader] = React.useState(true);
    let [roles, setRoles] = React.useState<IRoleItem[]>([]);
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SETTINGS_REQUESTS }));

    const userId = useCurrentUser()?.user_id;

    React.useEffect(() => {
        init();

        return () => {
            request.current.abort();
        };
    }, []);

    const init = () => {
        request.current
            .exec(REQUESTS.GET_ROLES)
            .then((response) => {
                let isOnlyOne = response?.report?.length === 1;
                setRoles(
                    response?.report
                        ?.sort((a, b) => {
                            return a.role_description?.role_id.localeCompare(b.role_description?.role_id);
                        })
                        ?.map((role) => {
                            return {
                                role_id: role.role_description?.role_id,
                                role_description: role.role_description?.role_description,
                                is_active: role.user_role?.active === '1',
                                user_id: role.user_role.user_id,
                                disabled: isOnlyOne || false,
                            };
                        }),
                );
            })
            .finally(() => {
                setLoader(false);
            });
    };

    const action = (roles, role_id, state) => {
        return roles.map((role) => {
            return role.role_id == role_id ? Object.assign({}, role, state) : role;
        });
    };

    const onChange = (role_id, user_id, state) => {
        if (roles.filter((role) => role.is_active).length == 1 && !state) {
            setRoles((roles) => action(roles, role_id, { disabled: false }));
        } else {
            setRoles((roles) => action(roles, role_id, { disabled: true }));

            request.current
                .exec(REQUESTS[state ? 'ACTIVATE' : 'DEACTIVATE'], {
                    queryParams: {
                        role_id,
                        user_id,
                    },
                })
                .then(() => {
                    setRoles((roles) =>
                        action(roles, role_id, {
                            is_active: state,
                            disabled: false,
                        }),
                    );
                })
                .catch(() => {
                    setRoles((roles) => action(roles, role_id, { disabled: false }));
                });
        }
    };

    const showSettingsTags = showSettingsTagsFlag();

    return (
        <ContentContainer bodyScroll>
            <Widget
                className={styles.widget}
                contentContainer
            >
                <div className={styles.wrap}>
                    <Header3 className={styles.header1}>{i18n('Manage roles')}</Header3>
                    {isSwitchEnv() && (
                        <>
                            <div className={styles.header4}>{i18n('Environments')}</div>
                            <div className={styles.header4_1}>{i18n('The page will be refreshed after switching')}</div>
                            <div className={styles.switch_env}>
                                <EnvSwitcher />
                            </div>

                            <div className={styles.divider} />
                            <div className={styles.header2}>{i18n('Roles')} </div>
                        </>
                    )}

                    {isLoading ? (
                        <div className={styles.spin}>
                            <Spin />
                        </div>
                    ) : (
                        ''
                    )}

                    <div>
                        {roles?.map((role) => {
                            return (
                                <RoleItem
                                    role={role}
                                    key={role.role_id}
                                    onChange={onChange.bind(null, role.role_id, role.user_id)}
                                />
                            );
                        })}
                    </div>

                    {showSettingsTags && (
                        <>
                            <div className={styles.divider} />
                            <div className={styles.header2}>{i18n('Organization affiliation tags')} </div>
                            <UserOrganizationTags userId={userId ?? ''} />
                        </>
                    )}
                </div>
            </Widget>
        </ContentContainer>
    );
}
