import React, { useContext, useEffect, useState } from 'react';
import { matchPath, useHistory, useLocation } from 'react-router-dom';

import FlagsContext from 'contexts/FlagsContext';

import { getCarDashboardFlag } from 'utils/getCarDashboardFlag';
import { getDashboardComponent } from 'utils/getDashboardComponent';
import { isShowUsersSection } from 'utils/isShowUsersSection';
import { showTaxiCompanyRides } from 'utils/showTaxiCompanyRides';

import { Path } from 'shared/consts/Path';

import { MenuItemType } from 'components/Content/types';
import { MenuItem } from 'components/SideMenu/MenuItems/MenuItem';

import style from 'components/SideMenu/MenuItems/index.css';

interface IMenuItem {
    type: MenuItemType;
    link: Path;
    exact?: boolean;
}

export const MenuItems = () => {
    const history = useHistory();
    const location = useLocation();
    const [selected, setSelected] = useState<string | null>(null);
    const [menuItems, setMenuItems] = useState<IMenuItem[]>([]);
    let flags = useContext(FlagsContext);

    useEffect(() => {
        let menuItems: IMenuItem[] = [
            {
                type: MenuItemType.CARS,
                link: Path.CARS,
            },
            {
                type: MenuItemType.SIGNAL,
                link: Path.SIGNALS,
            },
            {
                type: MenuItemType.REPORTS,
                link: Path.REPORTS,
            },
            ...(isShowUsersSection()
                ? [
                      {
                          type: MenuItemType.USERS,
                          link: Path.USERS,
                      },
                  ]
                : []),
            {
                type: MenuItemType.SETTINGS,
                link: Path.SETTINGS,
            },
        ];

        if (showTaxiCompanyRides()) {
            menuItems.splice(1, 0, {
                type: MenuItemType.RIDES,
                link: Path.RIDES,
            });
        }

        if (getDashboardComponent()) {
            menuItems.unshift({
                type: MenuItemType.PARK,
                link: Path.HOME,
                exact: true,
            });
        }

        if (getCarDashboardFlag()) {
            menuItems.unshift({
                type: MenuItemType.DASHBOARD,
                link: Path.DASHBOARD,
            });
        }

        setMenuItems(menuItems);
    }, [flags]);

    useEffect(() => {
        setSelected(null);

        menuItems?.forEach((menuItem) => {
            let match = matchPath(location.pathname, {
                path: menuItem.link,
                exact: menuItem?.exact ?? false,
            });

            if (match !== null) {
                setSelected(menuItem.link);
            }
        });
    }, [location, flags]);

    const onMenuItemClick = (link: string) => {
        setSelected(link);
        history.push(link);
    };

    return (
        <div className={style.menu}>
            {menuItems.map((menuItem) => (
                <MenuItem
                    key={menuItem.link}
                    type={menuItem.type}
                    link={menuItem.link}
                    onClick={onMenuItemClick}
                    selected={selected === menuItem.link}
                />
            ))}
        </div>
    );
};
