import React from 'react';

import { getTLD } from 'utils/getTLD';
import { isSwitchEnv } from 'utils/isSwitchEnv';
import { getLang } from 'utils/language/getLang';
import { setLang } from 'utils/language/setLang';

import { EnvSwitcher } from 'features/EnvSwitcher';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Button } from 'shared/ui/Button/Button';

import { IPassportUserInfo } from 'components/Content/types';
import { AccountInfo } from 'components/SideMenu/UserMenu/UserMenuWidget/AccountInfo';
import { IRadioButtonValues, RadioButton } from 'components/ui/RadioButton';
import Widget from 'components/ui/Widget';

import { i18n } from 'components/SideMenu/UserMenu/UserMenuWidget/component.i18n';

import LogoutIcon from 'components/SideMenu/UserMenu/UserMenuWidget/icons/logout.component.svg';

import styles from 'components/SideMenu/UserMenu/UserMenuWidget/index.css';

interface IUserMenuWidgetProps {
    userInfo: IPassportUserInfo | null;
    logOutClick: () => void;
}

const LANGS: IRadioButtonValues[] = [
    { value: 'en', placeholder: 'En' },
    ...(getTLD() === 'ru' ? [{ value: 'ru', placeholder: 'Ru' }] : []),
    { value: 'cs', placeholder: 'Cz' },
];

export const UserMenuWidget = React.forwardRef(
    (props: IUserMenuWidgetProps, ref: React.MutableRefObject<HTMLDivElement | null>) => {
        const { userInfo, logOutClick } = props;
        const lang = getLang();

        const onChangeHandler = React.useCallback((value) => {
            setLang(value);
            location.reload();
        }, []);

        return (
            <div
                className={styles.userMenuContainer}
                ref={ref}
            >
                <Widget
                    topLevel
                    contentComponent={
                        <div className={styles.userMenuWidget}>
                            <AccountInfo userInfo={userInfo} />

                            {isSwitchEnv() && (
                                <div className={styles.item}>
                                    <EnvSwitcher />
                                </div>
                            )}

                            <div className={styles.item}>
                                <RadioButton
                                    fullWidth
                                    secondary
                                    selectedValue={lang}
                                    values={LANGS}
                                    onChange={onChangeHandler}
                                />
                            </div>

                            <div className={styles.logout}>
                                <Button
                                    wide
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    icon={LogoutIcon}
                                    label={i18n('Log out')}
                                    onClick={logOutClick}
                                />
                            </div>
                        </div>
                    }
                />
            </div>
        );
    },
);
