import React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isAddNewSignal } from 'utils/isAddNewSignal';

import { ManageSignalsContentHeader } from 'widgets/ManageSignalsContentHeader/ui/ManageSignalsContentHeader/ManageSignalsContentHeader';

import { ManageSignalsFilter } from 'entities/Signal/consts/ManageSignalsFilter';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { Widget } from 'shared/ui/Widget/Widget';

import { ModalObjectTypes, StatusTypes } from 'components/types';
import DescriptionIcon, { Direction } from 'components/ui/DescriptionIcon';
import { Label, LabelIndicatorTypes } from 'components/ui/Label';
import { Status } from 'components/ui/Status';
import Table from 'components/ui/Table';
import { ITableRowData } from 'components/ui/Table/types';

import { i18n } from 'components/Signals/ManageSignals/ManageSignalsContent/index.i18n';

import style from 'components/Signals/ManageSignals/ManageSignalsContent/index.css';

const ManageSignalsContent = ({ signals_descriptions, filters, isLoading, error, reloadFunction }) => {
    let location = useLocation();
    let history = useHistory();

    const SIGNALS_TABLE_HEADER = [
        { key: 'name', displayName: i18n('Signal') },
        { key: 'state', displayName: i18n('State') },
        { key: 'level', displayName: i18n('Level') },
    ];

    const statuses = {
        critical: i18n('Critical'),
        normal: i18n('Normal'),
    };

    const onRowClick = (signal: string) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SIGNAL_SETTINGS);
        searchParams.set(MODAL_OBJECT_ID_CGI, signal);
        history.push(`${location.pathname}?${searchParams}`);
    };

    const buildTableRows = (): ITableRowData[] => {
        return signals_descriptions
            ?.filter((signal) => {
                return (
                    !filters[ManageSignalsFilter.PRIORITY] ||
                    filters[ManageSignalsFilter.PRIORITY] === 'all' ||
                    filters[ManageSignalsFilter.PRIORITY].includes(signal.priority)
                );
            })
            .map((signal) => {
                let isCritical = signal.priority === 'critical';

                return {
                    data: {
                        name: (
                            <div
                                className={`${style.signal_name} ${
                                    !signal?.is_editable ? style.not_editable_name : ''
                                }`}
                            >
                                {signal.display_name}
                                {!signal?.is_editable ? (
                                    <DescriptionIcon
                                        description={i18n('Basic signals cannot be edited')}
                                        small
                                        direction={Direction.TOP}
                                    />
                                ) : null}
                            </div>
                        ),

                        state: signal?.is_enabled ? (
                            <Status
                                statusType={StatusTypes.RIDING}
                                big
                                customLabel={i18n('Enabled')}
                            />
                        ) : (
                            <Status
                                statusType={StatusTypes.DELAYED_BLOCK}
                                big
                                customLabel={i18n('Disabled')}
                            />
                        ),

                        level: (
                            <Label
                                background
                                indicator={isCritical ? { type: LabelIndicatorTypes.ERROR } : true}
                                title={statuses[signal.priority]}
                            />
                        ),
                    },

                    meta: {
                        key: signal.name,
                        className: isAddNewSignal() && signal?.is_editable ? style.signal_row : '',
                        onClick: isAddNewSignal() && signal?.is_editable ? onRowClick.bind(null, signal.name) : null,
                    },
                };
            });
    };

    return (
        <ContentContainer
            bodyScroll
            withSidebar
        >
            <Widget contentContainer>
                {isAddNewSignal() && <ManageSignalsContentHeader />}

                <Table
                    header={SIGNALS_TABLE_HEADER}
                    error={error}
                    isLoading={isLoading}
                    reloadFunction={reloadFunction}
                    tableData={buildTableRows()}
                />
            </Widget>
        </ContentContainer>
    );
};

export default ManageSignalsContent;
