import React, { MutableRefObject, useRef, useState } from 'react';
import { useLocation } from 'react-router-dom';

import { MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { ManageSignalsFilters } from 'widgets/ManageSignalsFilters';

import { SignalsHeaderTabs } from 'features/SignalsHeaderTabs';

import { getCarSignalDescriptionDisplayName } from 'entities/Car/helpers/getCarSignalDescriptionDisplayName/getCarSignalDescriptionDisplayName';
import { INITIAL_MANAGE_SIGNALS_FILTERS } from 'entities/Signal/consts/filters';
import { ManageSignalsFilter } from 'entities/Signal/consts/ManageSignalsFilter';
import { SignalDescriptionSchema } from 'entities/Signal/types/SignalDescriptionSchema';

import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import ManageSignalsContent from 'components/Signals/ManageSignals/ManageSignalsContent';
import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import { ISignalDescriptionValue } from 'components/Signals/types';

import { RequestHelper } from '../../../../request-helper/src';

import { i18n } from 'components/Signals/ManageSignals/index.i18n';

export interface ManageSignalsFiltersOptions {
    [ManageSignalsFilter.PRIORITY]: Nullable<string>;
}

const ManageSignals = () => {
    const location = useLocation();

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));

    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [signals_descriptions, setData] = useState<any[] | null>(null);
    const [isGlobalSidebarOpen, setIsGlobalSidebarOpen] = useState<boolean>(false);
    const [filters, setFilters] = React.useState<ManageSignalsFiltersOptions>(INITIAL_MANAGE_SIGNALS_FILTERS);

    React.useEffect(() => {
        getDescriptions();

        return () => {
            request?.current?.abort();
        };
    }, []);

    React.useEffect(() => {
        let searchParams = new URLSearchParams(location.search);
        let isGlobalSidebarOpenSearch = searchParams.get(MODAL_OBJECT_TYPE_CGI);
        if (isGlobalSidebarOpen !== !!isGlobalSidebarOpenSearch) {
            if (!isGlobalSidebarOpenSearch) {
                //when sidebar is closed, update descriptions
                getDescriptions();
            }
            setIsGlobalSidebarOpen(!!isGlobalSidebarOpenSearch);
        }
    }, [location]);

    const getDescriptions = () => {
        setIsLoading(true);
        setError(null);
        request.current
            .exec(REQUESTS.GET_SIGNALS_DESCRIPTIONS)
            .then((response) => {
                setIsLoading(false);
                setError(null);
                response?.signals_descriptions && setData(transformData(response.signals_descriptions));
            })
            .catch((error) => {
                setIsLoading(false);
                setError(error);
            });
    };
    const transformData = (data: Record<string, ISignalDescriptionValue>): ISignalDescriptionValue[] => {
        return Object.entries(data).map(([, el]) => {
            return {
                display_name: getCarSignalDescriptionDisplayName(el as SignalDescriptionSchema),
                name: el.name,
                priority: el.priority,
                is_enabled: el.is_enabled,
                is_editable: el.is_editable,
            };
        });
    };

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Signal Center')}
                    tabs={<SignalsHeaderTabs />}
                    withoutBorder
                />
            }
            filters={
                <ManageSignalsFilters
                    offsetTop={HEADER_HEIGHT}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ManageSignalsContent
                isLoading={isLoading}
                reloadFunction={getDescriptions}
                error={error}
                signals_descriptions={signals_descriptions}
                filters={filters}
            />
        </SectionLayout>
    );
};

export default ManageSignals;
