import React from 'react';
import { useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { durationUTCBetween } from 'utils/date/durationUTCBetween';
import shortDateTime from 'utils/date/shortDateTime';

import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { CarNumberLabel } from 'components/Cars/CarNumberLabel';
import { SCORING_TRACE_TAG } from 'components/Cars/constants';
import { FormattedSignal, TableHeadersKeys } from 'components/Signals/types';
import { ModalObjectTypes } from 'components/types';
import { DurationUTCBetween } from 'components/ui/DurationUTCBetween';
import { Label, LabelIndicatorTypes } from 'components/ui/Label';
import Table from 'components/ui/Table';
import TableLink from 'components/ui/Table/TableLink';
import { ITableGetMore, ITableRowData } from 'components/ui/Table/types';

import { i18n } from 'components/Signals/SignalsTable/index.i18n';

import style from 'components/Signals/SignalsTable/index.css';

export interface IEntityObject {
    car: { object_id: string; details?: any } | null;
    trace: { object_id: string; details?: any } | null;
}

interface ISignalsTableProps {
    isLoading: boolean;
    error: Error | null;
    signals: FormattedSignal[];
    getSignals: () => void;
    drawColumns?: TableHeadersKeys[];
    getMore?: ITableGetMore | null;
}

const signalsTranslates = {
    'Блокировка машины': i18n('Car is blocked'),
    'Лаг телематики': i18n('Telematics lag'),
    'Опасная поездка': i18n('Dangerous ride'),
    'Очень опасная поездка': i18n('Very dangerous ride'),
    'Выезд из зоны': i18n('Leaving geofence'),
    'Простой: 1 час': i18n('Inactivity: 1 hour'),
};

export const SIGNALS_TABLE_HEADER: {
    key: TableHeadersKeys;
    displayName: string;
}[] = [
    { key: 'date_time', displayName: i18n('Date and time') },
    { key: 'event', displayName: i18n('Signal') },
    { key: 'car', displayName: i18n('Car') },
    { key: 'session', displayName: i18n('Ride') },
    { key: 'duration', displayName: i18n('Duration') },
];

export const SIGNALS_TABLE_REDESIGN_HEADER: {
    key: TableHeadersKeys;
    displayName: string;
}[] = [
    { key: 'date_time', displayName: i18n('Date and time') },
    { key: 'level', displayName: i18n('Level') },
    { key: 'event', displayName: i18n('Signal') },
    { key: 'car', displayName: i18n('Car') },
    { key: 'session', displayName: i18n('Ride') },
    { key: 'duration', displayName: i18n('Duration') },
];

const SignalsTable = ({ isLoading, error, signals, getSignals, getMore, drawColumns }: ISignalsTableProps) => {
    let location = useLocation();
    const SIGNALS_TABLE_DISPLAY_HEADER = drawColumns
        ? SIGNALS_TABLE_HEADER.filter((headerItem) => drawColumns.includes(headerItem.key))
        : SIGNALS_TABLE_HEADER;

    const buildTableRows = (): ITableRowData[] => {
        return signals.map((signal) => {
            let { since, until, name, description, linked_entities, is_actual } = signal;

            let displayName = description?.display_name ?? name;

            let key = since.toString();
            let entitiesObject: IEntityObject = linked_entities.reduce(
                (result, linkedEntity) => {
                    let { entity_type, object_id } = linkedEntity;
                    key += `_${object_id}`;
                    result[entity_type] = linkedEntity;

                    return result;
                },
                { car: null, trace: null },
            );

            let time = since ? shortDateTime(since) : EMPTY_DATA;

            let trace = entitiesObject.trace ?? null;
            let traceLink = '';
            if (trace) {
                let searchParams = new URLSearchParams(location.search);
                searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
                searchParams.set(MODAL_OBJECT_ID_CGI, trace.object_id);
                searchParams.set('back', `${location.pathname}?${location.search}`);

                traceLink = `${location.pathname}?${searchParams}`;
            }

            return {
                data: {
                    date_time: (
                        <div className={style.since}>
                            {time} {is_actual ? <div className={style.is_actual_icon} /> : null}
                        </div>
                    ),

                    duration:
                        name !== SCORING_TRACE_TAG ? (
                            <DurationUTCBetween duration={durationUTCBetween(since, until)} />
                        ) : null,
                    event: (
                        <Label
                            background
                            indicator={
                                description?.priority === 'critical'
                                    ? { type: LabelIndicatorTypes.ERROR }
                                    : description?.priority === 'warning'
                                    ? { type: LabelIndicatorTypes.WARNING }
                                    : true
                            }
                            title={signalsTranslates[displayName as any] || displayName}
                        />
                    ),

                    car: entitiesObject.car ? (
                        <TableLink
                            key={entitiesObject.car?.object_id}
                            link={`/cars/${entitiesObject.car?.object_id}`}
                        >
                            <CarNumberLabel
                                carInfo={entitiesObject.car?.details}
                                onlyText
                                shortVIN
                            />
                        </TableLink>
                    ) : null,
                    session: entitiesObject.trace ? (
                        <TableLink
                            key={entitiesObject.trace?.object_id}
                            link={traceLink}
                        >
                            <SessionTime
                                oneRow
                                key={entitiesObject.trace?.object_id}
                                start={entitiesObject.trace?.details?.since}
                                finish={entitiesObject.trace?.details?.until}
                            />
                        </TableLink>
                    ) : null,
                },

                meta: {
                    key: key,
                },
            };
        });
    };

    return (
        <Table
            header={SIGNALS_TABLE_DISPLAY_HEADER}
            reloadFunction={getSignals}
            tableData={buildTableRows()}
            getMore={getMore}
            isLoading={isLoading}
            error={error}
        />
    );
};

export default SignalsTable;
