import React, { MutableRefObject, useEffect, useRef, useState } from 'react';

import {
    ALL_SIGNALS_UNCHECKED_KEY,
    DEBOUNCE_TIMEOUT,
    SIGNALS_PAGE_SIZE,
    VERY_DANGEROUS_SESSION_BORDER,
} from 'constants/constants';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { SignalsContentHeader } from 'widgets/SignalsContentHeader';
import { SignalsFilters } from 'widgets/SignalsFilters';

import { FilterDateOptions } from 'features/FilterDate/ui/FilterDate/FilterDate';
import { SignalsHeaderTabs } from 'features/SignalsHeaderTabs';

import { INITIAL_SIGNALS_FILTERS } from 'entities/Signal/consts/filters';
import { SignalsFilter } from 'entities/Signal/consts/SignalsFilter';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import { REQUESTS, SIGNALS_REQUESTS } from 'components/Signals/request';
import SignalsTable from 'components/Signals/SignalsTable';
import { FormattedSignal, ISignalsResponse } from 'components/Signals/types';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Signals/index.i18n';

export interface SignalsFiltersOptions extends FilterDateOptions {
    [SignalsFilter.SIGNALS]: Nullable<string>;
    [SignalsFilter.CARS_IDS]?: Nullable<string>;
    [SignalsFilter.ACTUAL_ONLY]: Nullable<boolean>;
}

const Signals = () => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: SIGNALS_REQUESTS }));
    const updateTimer: MutableRefObject<any> = useRef(null);

    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [formattedSignals, setFormattedSignals] = useState<FormattedSignal[]>([]);
    const [filters, setFilters] = useState<SignalsFiltersOptions>(INITIAL_SIGNALS_FILTERS);
    const [canGetMore, setCanGetMore] = useState<boolean>(false);
    const [isMoreLoading, setIsMoreLoading] = useState<boolean>(true);
    const [nextCarsCursor, setNextCarsCursor] = useState<number | null>(null);
    const [nextSessionsCursor, setNextSessionsCursor] = useState<number | null>(null);

    useEffect(() => {
        return () => {
            request.current.abort();
            clearTimeout(updateTimer.current);
        };
    }, []);

    useEffect(() => {
        if (filters) {
            clearTimeout(updateTimer.current);
            updateTimer.current = setTimeout(() => {
                getSignals();
            }, DEBOUNCE_TIMEOUT);
        }
    }, [filters]);

    const fetchSignals = ({
        nextCarsCursor,
        nextSessionsCursor,
        isMore,
        pageSize,
    }: {
        nextCarsCursor: number | null;
        nextSessionsCursor: number | null;
        pageSize?: number;
        isMore?: boolean;
    }): Promise<ISignalsResponse & { formattedSignalsRes: FormattedSignal[] }> => {
        request.current.abort();

        return request.current
            .exec(REQUESTS.GET_SIGNALS, {
                queryParams: {
                    since: filters.since ?? null,
                    until: filters.until ? Math.trunc(filters.until) : null,
                    actual_only: filters.actual_only ?? null,
                    signals: filters.signals
                        ? filters.signals !== ALL_SIGNALS_UNCHECKED_KEY
                            ? filters.signals
                            : null
                        : null,
                    page_size: pageSize,
                    cars_cursor: isMore ? nextCarsCursor : null,
                    sessions_cursor: isMore ? nextSessionsCursor : null,
                },
            })
            .then((signalsResponse: ISignalsResponse) => {
                let formattedSignalsRes =
                    signalsResponse.signals?.map((signal) => {
                        let { name, details } = signal;
                        let description = { ...(signalsResponse.signals_descriptions?.[name] || {}) };
                        let formattedSignal: FormattedSignal = Object.assign({}, { description }, signal);

                        if (name === 'scoring_trace_tag' && formattedSignal?.description) {
                            let sessionScore = details?.score ?? 0;
                            formattedSignal.description.display_name =
                                sessionScore < VERY_DANGEROUS_SESSION_BORDER
                                    ? 'Опасная поездка'
                                    : 'Очень опасная поездка';
                            formattedSignal.description.priority =
                                sessionScore < VERY_DANGEROUS_SESSION_BORDER ? 'warning' : 'critical';
                        }

                        return formattedSignal;
                    }) ?? [];

                return { ...signalsResponse, formattedSignalsRes };
            });
    };

    const getSignals = (isMore?: boolean) => {
        if (isMore) {
            setIsMoreLoading(true);
        } else {
            setIsLoading(true);
        }

        fetchSignals({
            nextCarsCursor,
            nextSessionsCursor,
            isMore,
            pageSize: SIGNALS_PAGE_SIZE,
        })
            .then((signalsResponse) => {
                let { formattedSignalsRes, can_get_more_pages, next_cars_cursor, next_sessions_cursor } =
                    signalsResponse;

                setError(null);
                setIsMoreLoading(signalsResponse.meta === ABORT_ERROR_KEY);
                setIsLoading(signalsResponse.meta === ABORT_ERROR_KEY);
                setCanGetMore(can_get_more_pages ?? false);
                setNextCarsCursor(next_cars_cursor ?? null);
                setNextSessionsCursor(next_sessions_cursor ?? null);

                setFormattedSignals(isMore ? [...formattedSignals, ...formattedSignalsRes] : formattedSignalsRes);
            })
            .catch((error) => {
                setIsMoreLoading(false);
                setIsLoading(false);
                setError(error);
            });
    };

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Signal Center')}
                    tabs={<SignalsHeaderTabs />}
                    withoutBorder
                />
            }
            filters={
                <SignalsFilters
                    offsetTop={HEADER_HEIGHT}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ContentContainer
                bodyScroll
                withSidebar
            >
                <Widget contentContainer>
                    <SignalsContentHeader filters={filters} />

                    <SignalsTable
                        getSignals={getSignals}
                        isLoading={isLoading}
                        error={error}
                        getMore={{
                            canGetMore,
                            isMoreLoading,
                            onGetMoreClick: getSignals.bind(null, true),
                        }}
                        signals={formattedSignals}
                    />
                </Widget>
            </ContentContainer>
        </SectionLayout>
    );
};

export default Signals;
