import React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isUserProfilePageFlag } from 'utils/isUserProfilePageFlag';
import { showUserScoringInformer } from 'utils/showUserScoringInformer';

import { UsersContentHeader } from 'widgets/UsersContentHeader';

import { getUserRoles, UserStatusInfo } from 'entities/User';
import { UserRole } from 'entities/User/consts/UserRole';
import { UserStatus } from 'entities/User/consts/UserStatus';

import { Path } from 'shared/consts/Path';
import { formatPhoneNumber } from 'shared/helpers/formatPhoneNumber/formatPhoneNumber';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { Scoring } from 'shared/ui/Scoring/Scoring';
import { Widget } from 'shared/ui/Widget/Widget';

import { ModalObjectTypes } from 'components/types';
import Table from 'components/ui/Table';
import { IUser } from 'components/Users/types';

import { i18n } from 'components/Users/UsersContent/index.i18n';

import style from 'components/Users/UsersContent/index.css';

const MAX_ROLES_LABELS_IN_ROW = 2;
const INACTIVE_STATUS = 'inactive';

interface IUsersContentProps {
    isLoading: boolean;
    isMoreLoading: boolean;
    hasMore: boolean;
    error: Error | null;
    users: IUser[];
    getUsers?: () => void;
}

interface TableDataResult {
    name: React.ReactNode;
    status: React.ReactNode;
    scoring?: React.ReactNode;
    roles: React.ReactNode;
    phone: React.ReactNode;
    email: React.ReactNode;
}

const UsersContent = ({ isLoading, isMoreLoading, hasMore, error, users, getUsers }: IUsersContentProps) => {
    const location = useLocation();
    const history = useHistory();

    const isScoringVisible = showUserScoringInformer();

    const header = React.useMemo(() => {
        let array = [
            { key: 'name', displayName: i18n('Name') },
            { key: 'status', displayName: i18n('Status') },
            { key: 'roles', displayName: i18n('Roles') },
            { key: 'phone', displayName: i18n('Phone') },
            { key: 'email', displayName: i18n('E-mail') },
        ];

        if (isScoringVisible) {
            array.splice(2, 0, { key: 'scoring', displayName: i18n('Scoring') });
        }

        return array;
    }, [isScoringVisible]);

    const onRowClick = React.useCallback((userId: string) => {
        if (isUserProfilePageFlag()) {
            history.push(`${Path.USERS}/${userId}/profile`);
        } else {
            const searchParams = new URLSearchParams(location.search);

            searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.USER);
            searchParams.set(MODAL_OBJECT_ID_CGI, userId);
            history.push(`${location.pathname}?${searchParams}`);
        }
    }, []);

    const textCell = React.useCallback((value: Optional<string>) => {
        return (
            <div
                className={style.text_cell}
                title={value || EMPTY_DATA}
            >
                {value || EMPTY_DATA}
            </div>
        );
    }, []);

    const tableData = React.useMemo(() => {
        return users.map((user, index) => {
            let { id, last_name, first_name, pn, phone_number, email, status, roles, user_score } = user;

            let displayName = [last_name, first_name, pn].join(' ');

            let resultData: TableDataResult = {
                name: textCell(displayName),
                status: <UserStatusInfo status={status as UserStatus} />,
                roles: roles?.length ? (
                    <>
                        {getUserRoles(roles.slice(0, MAX_ROLES_LABELS_IN_ROW) as UserRole[]).join(', ')}

                        {roles.length > MAX_ROLES_LABELS_IN_ROW ? (
                            <span className={style.more_label}>
                                {i18n('{count} more', {
                                    count: roles.length - MAX_ROLES_LABELS_IN_ROW,
                                })}
                            </span>
                        ) : null}
                    </>
                ) : (
                    EMPTY_DATA
                ),

                phone: textCell(formatPhoneNumber(phone_number)),
                email: textCell(email),
            };

            if (isScoringVisible) {
                resultData.scoring = <Scoring value={user_score?.rank} />;
            }

            return {
                data: resultData,
                meta: {
                    key: id || index.toString(),
                    className: `${style.user_row} ${status === INACTIVE_STATUS ? style.inactive : ''}`,
                    onClick: onRowClick.bind(null, id),
                },
            };
        });
    }, [users, onRowClick]);

    return (
        <ContentContainer>
            <Widget contentContainer>
                <UsersContentHeader onUserCreate={getUsers} />

                <Table
                    header={header}
                    tableData={tableData}
                    error={error}
                    isLoading={isLoading}
                    reloadFunction={getUsers}
                    getMore={{
                        canGetMore: hasMore,
                        onGetMoreClick: getUsers?.bind(null, true),
                        isMoreLoading: isMoreLoading,
                    }}
                />
            </Widget>
        </ContentContainer>
    );
};

export default UsersContent;
