import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useLocation } from 'react-router-dom';

import { MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { Header } from 'widgets/Header';

import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import { REQUESTS, USERS_REQUESTS } from 'components/Users/request';
import { IUser } from 'components/Users/types';
import UsersContent from 'components/Users/UsersContent';

import { ABORT_ERROR_KEY, RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Users/index.i18n';

const Users = () => {
    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: USERS_REQUESTS }));
    const [users, setUsers] = useState<IUser[]>([]);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [isMoreLoading, setMoreIsLoading] = useState<boolean>(true);
    const [hasMore, setHasMore] = useState<boolean>(false);
    const [error, setError] = useState<Error | null>(null);
    const [isGlobalSidebarOpen, setIsGlobalSidebarOpen] = useState<boolean>(false);
    let location = useLocation();

    useEffect(() => {
        getUsers();

        return () => request.current.abort();
    }, []);

    useEffect(() => {
        let searchParams = new URLSearchParams(location.search);
        let isGlobalSidebarOpenSearch = searchParams.get(MODAL_OBJECT_TYPE_CGI);
        if (isGlobalSidebarOpen !== !!isGlobalSidebarOpenSearch) {
            if (!isGlobalSidebarOpenSearch) {
                //when sidebar is closed, update descriptions
                getUsers();
            }
            setIsGlobalSidebarOpen(!!isGlobalSidebarOpenSearch);
        }
    }, [location]);

    const getUsers = React.useCallback((more?: boolean) => {
        if (more) {
            setMoreIsLoading(true);
        } else {
            setIsLoading(true);
        }
        setError(null);
        request.current.abort();

        request.current
            .exec(REQUESTS.GET_USERS)
            .then((response) => {
                let usersResponse = response.members ?? [];
                let hasMore = response.has_more ?? false;
                setUsers(more ? [...users, ...usersResponse] : usersResponse);
                setIsLoading(response?.meta === ABORT_ERROR_KEY);
                setMoreIsLoading(response?.meta === ABORT_ERROR_KEY);
                setHasMore(hasMore);
            })
            .catch((error) => {
                setError(error);
                setIsLoading(false);
                setMoreIsLoading(false);
            });
    }, []);

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Users')}
                    withoutBorder
                />
            }
            bodyScroll
        >
            <UsersContent
                users={users}
                isLoading={isLoading}
                isMoreLoading={isMoreLoading}
                hasMore={hasMore}
                error={error}
                getUsers={getUsers}
            />
        </SectionLayout>
    );
};

export default Users;
