import React, { MutableRefObject, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import ControlButton from 'components/ui/Buttons/ControlButton';
import Confirm from 'components/ui/Confirm';
import { Input } from 'components/ui/Input';
import SectionFiltersSidebar from 'components/ui/SectionLayout/SectionFiltersSidebar';
import MainText3 from 'components/ui/Text/MainText3';
import Subhead1 from 'components/ui/Text/Subhead1';
import { REQUESTS, ZONES_REQUESTS } from 'components/Zones/request';
import { IArea, SELECTED_ZONE_CGI } from 'components/Zones/types';

import { RequestHelper } from '../../../../../request-helper/src';

import { i18n } from 'components/Zones/ZonesSidebar/ZoneSidebar/index.i18n';

import BackArrow from 'components/Zones/ZonesSidebar/ZoneSidebar/back_arrow.component.svg';

import style from 'components/Zones/ZonesSidebar/ZoneSidebar/index.css';

interface IZoneSidebarProps {
    onBackClick: () => void;
    onCreateZoneClick: () => void;
    isCreateZoneInfoFilled: boolean;
    onCreateZoneFormChange: ({ zoneName: string }) => void;
    isCreatingZone: boolean;
    creatingZoneError: Error | null;
    isLoading: boolean;
    selectedZone?: string | null;
    zone?: IArea[] | null;
    onRemoveZone: () => void;
}

const ZoneSidebar = ({
    onBackClick: onBackClickProps,
    onCreateZoneClick: onCreateZoneClickProps,
    isCreatingZone,
    onCreateZoneFormChange,
    isCreateZoneInfoFilled,
    selectedZone,
    zone,
    isLoading,
    onRemoveZone: onRemoveZoneProps,
}: IZoneSidebarProps) => {
    const [zoneName, setZoneName] = useState<string>('');
    const [isDeleteConfirmOpen, setIsDeleteConfirmOpen] = useState<boolean>(false);

    const [isZoneInfoChanged, setIsZoneInfoChanged] = useState<boolean>(false);

    const [isRemoving, setIsRemoving] = useState<boolean>(false);
    const [removingError, setRemovingError] = useState<Error | null>(null);

    const [isBackConfirmOpen, setIsBackConfirmOpen] = useState<boolean>(false);

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: ZONES_REQUESTS }));

    let location = useLocation();
    let history = useHistory();

    useEffect(() => {
        if (selectedZone && zone) {
            setZoneName(selectedZone);
            setIsZoneInfoChanged(false);
        }
    }, [zone]);

    useEffect(() => {
        let isZoneInfoChanged = zone ? selectedZone !== zoneName : zoneName !== '';
        onCreateZoneFormChange({ zoneName });
        setIsZoneInfoChanged(isZoneInfoChanged);
    }, [zoneName]);

    const openDeleteConfirm = () => {
        setIsDeleteConfirmOpen(true);
    };

    const closeDeleteConfirm = () => {
        setIsDeleteConfirmOpen(false);
        setIsRemoving(false);
        setRemovingError(null);
    };

    const onDeleteConfirm = () => {
        if (selectedZone) {
            onRemoveZone(selectedZone);
        }
    };

    const onBackClick = () => {
        if (isZoneInfoChanged) {
            openBackConfirm();
        } else {
            onBackClickProps();
        }
    };

    const openBackConfirm = () => {
        setIsBackConfirmOpen(true);
    };

    const closeBackConfirm = () => {
        setIsBackConfirmOpen(false);
    };

    const onCreateZoneClick = () => {
        onCreateZoneClickProps();
    };

    const onRemoveZone = (zoneName: string) => {
        setIsRemoving(true);
        setRemovingError(null);

        request.current
            .exec(REQUESTS.REMOVE_ZONE, { body: { ids: [zoneName] } })
            .finally(() => {
                setIsRemoving(false);
            })
            .then(() => {
                let searchParams = new URLSearchParams(location.search);
                searchParams.delete(SELECTED_ZONE_CGI);
                history.push(`${location.pathname}?${searchParams}`);
                onRemoveZoneProps();
            })
            .catch(setRemovingError);
    };

    return (
        <>
            <div
                className={style.back_button}
                onClick={onBackClick}
            >
                <BackArrow />
                <Subhead1>{i18n('Back')}</Subhead1>
            </div>
            <SectionFiltersSidebar
                filtersControls={[
                    {
                        filterComponent: (
                            <Input
                                disabled={isLoading || !!selectedZone}
                                required
                                label={i18n('Geofence name')}
                                value={zoneName}
                                onChange={setZoneName}
                            />
                        ),
                    },

                    ...(zone
                        ? [
                              {
                                  filterComponent: (
                                      <div
                                          className={style.delete_button}
                                          onClick={openDeleteConfirm}
                                      >
                                          <Subhead1 className={style.title}>{i18n('Delete geofence')}</Subhead1>
                                          <MainText3 className={style.description}>
                                              {i18n('All signals will lose their connection to this zone.')}
                                          </MainText3>
                                      </div>
                                  ),
                              },
                          ]
                        : []),
                ]}
                processResultControls={
                    isLoading || (zone && !isZoneInfoChanged)
                        ? []
                        : [
                              <ControlButton
                                  key="save_zone_button"
                                  fullWidth
                                  disabled={!isCreateZoneInfoFilled || isCreatingZone}
                                  title={zone ? i18n('Save changes') : i18n('Create geofence')}
                                  onClick={onCreateZoneClick}
                              />,
                          ]
                }
            />

            {isDeleteConfirmOpen ? (
                <Confirm
                    title={`${i18n('Delete geofence')}?`}
                    description={i18n('All signals will lose their connection to this zone.')}
                    onConfirm={onDeleteConfirm}
                    onClose={closeDeleteConfirm}
                    error={removingError}
                    isLoading={isRemoving}
                />
            ) : null}
            {isBackConfirmOpen ? (
                <Confirm
                    title={i18n('Unsaved changes')}
                    description={i18n(
                        'Are you sure you want to exit without saving the zone? All current changes will be lost.',
                    )}
                    confirmButtonTitle={i18n('Yes, go back')}
                    onConfirm={onBackClickProps}
                    onClose={closeBackConfirm}
                />
            ) : null}
        </>
    );
};

export default ZoneSidebar;
