import React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import ControlButton, { ButtonSize } from 'components/ui/Buttons/ControlButton';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import SectionFiltersSidebar from 'components/ui/SectionLayout/SectionFiltersSidebar';
import SectionSidebarTabs from 'components/ui/SectionLayout/SectionSidebarTabs';
import { IArea, SELECTED_ZONE_CGI } from 'components/Zones/types';
import ZoneSidebar from 'components/Zones/ZonesSidebar/ZoneSidebar';

import { i18n } from 'components/Zones/ZonesSidebar/index.i18n';

import Icon from 'components/ui/icons/add_S.component.svg';

import coreStyle from 'components/Content/index.css';
import style from 'components/Zones/ZonesSidebar/index.css';

const SHIMMERS_COUNT = 4;

interface IZonesSidebarProps {
    isLoading: boolean;
    error: Error | null;
    zones: { [key: string]: IArea[] } | null;
    selectedZone: string | null;
    isCreateZoneMode: boolean;
    getZones: () => void;
    createZone: () => void;
    onCreateZone: () => void;
    isCreatingZone: boolean;
    creatingZoneError: Error | null;
    onCreateZoneFormChange: ({ zoneName: string }) => void;
    isCreateZoneInfoFilled: boolean;
    onRemoveZone: () => void;
    stopCreateZone: () => void;
}

const ZonesSidebar = ({
    isLoading,
    error,
    zones,
    selectedZone,
    isCreateZoneMode,
    getZones,
    createZone,
    isCreatingZone,
    creatingZoneError,
    onCreateZone,
    isCreateZoneInfoFilled,
    onCreateZoneFormChange,
    onRemoveZone,
    stopCreateZone,
}: IZonesSidebarProps) => {
    let location = useLocation();
    let history = useHistory();

    const onTabClick = (zoneTitle: string) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(SELECTED_ZONE_CGI, zoneTitle);
        history.push(`${location.pathname}?${searchParams}`);
    };

    const onBackClick = () => {
        if (isCreateZoneMode) {
            stopCreateZone();
        } else {
            let searchParams = new URLSearchParams(location.search);
            searchParams.delete(SELECTED_ZONE_CGI);
            history.push(`${location.pathname}?${searchParams}`);
        }
    };

    let content: JSX.Element;
    if (selectedZone !== null || isCreateZoneMode) {
        content = (
            <ZoneSidebar
                onBackClick={onBackClick}
                onCreateZoneClick={onCreateZone}
                isCreatingZone={isCreatingZone}
                creatingZoneError={creatingZoneError}
                isCreateZoneInfoFilled={isCreateZoneInfoFilled}
                onCreateZoneFormChange={onCreateZoneFormChange}
                isLoading={isLoading}
                selectedZone={selectedZone}
                zone={selectedZone ? zones?.[selectedZone] : null}
                onRemoveZone={onRemoveZone}
            />
        );
    } else {
        content = (
            <>
                <SectionFiltersSidebar
                    sectionControls={[
                        {
                            component: (
                                <ControlButton
                                    size={ButtonSize.M}
                                    leftAlign
                                    tertiary
                                    fullWidth
                                    title={
                                        <>
                                            <Icon />
                                            {i18n('Create geofence')}
                                        </>
                                    }
                                    onClick={createZone}
                                />
                            ),
                        },
                    ]}
                />

                {isLoading ? (
                    <div className={style.shimmers_container}>
                        {new Array(SHIMMERS_COUNT).fill(true).map((_, index) => {
                            return (
                                <div
                                    key={index}
                                    className={style.shimmer_item}
                                >
                                    <div className={`${coreStyle.shimmer} ${style.big}`} />
                                    <div className={`${coreStyle.shimmer} ${style.small}`} />
                                </div>
                            );
                        })}
                    </div>
                ) : error ? (
                    <ErrorReloadLabel
                        title={i18n('Failed to load geofences')}
                        reloadFunction={getZones}
                    />
                ) : zones && Object.entries(zones)?.length ? (
                    <div className={style.tabs_container}>
                        <SectionSidebarTabs
                            tabs={Object.entries(zones).map(([zoneId]) => {
                                return {
                                    title: zoneId,
                                    onClick: onTabClick.bind(null, zoneId),
                                };
                            })}
                        />
                    </div>
                ) : (
                    <ErrorReloadLabel
                        title={i18n('There are currently no geofences')}
                        advice={i18n(
                            'Geofence are needed to set up leaving or entry signals and control movement of drivers',
                        )}
                        withoutIcon
                        reloadTitle={i18n('Create geofence')}
                        reloadFunction={createZone}
                    />
                )}
            </>
        );
    }

    return content;
};

export default ZonesSidebar;
