import React, { MutableRefObject, useContext, useEffect, useRef, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { MAP_DEFAULT_LAT, MAP_DEFAULT_LON, MAP_DEFAULT_ZOOM } from 'constants/constants';

import { getCenter } from 'utils/getCenter';

import { Header } from 'widgets/Header';

import { SignalsHeaderTabs } from 'features/SignalsHeaderTabs';

import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import ZoneEditor from 'components/Map/ZoneEditor';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { REQUESTS, ZONES_REQUESTS } from 'components/Zones/request';
import { IArea, SELECTED_ZONE_CGI } from 'components/Zones/types';
import ZonesSidebar from 'components/Zones/ZonesSidebar';

import { RequestHelper } from '../../../request-helper/src';

import { i18n } from 'components/Zones/index.i18n';

const DEFAULT_ZOOM = MAP_DEFAULT_ZOOM;
const BACKEND_CASH_REFRESH_TIMEOUT = 1000;

const Zones = () => {
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const [error, setError] = useState<Error | null>(null);
    const [zones, setZones] = useState<{ [key: string]: IArea[] } | null>(null);
    const [selectedZone, setSelectedZone] = useState<string | null>(null);

    const [isCreateZoneMode, setIsCreateZoneMode] = useState<boolean>(false);
    const [isCreatingZone, setIsCreatingZone] = useState<boolean>(false);
    const [creatingZoneError, setCreatingZoneError] = useState<Error | null>(null);

    const [creatingAreasCoordinates, setCreatingAreasCoordinates] = useState<[number, number][][]>([]);
    const [createZoneForm, setCreateZoneForm] = useState<{ zoneName: string }>({ zoneName: '' });
    const [isCreateZoneInfoFilled, setIsCreateZoneInfoFilled] = useState<boolean>(false);

    const [mapOptions, setMapOptions] = useState<any>({
        zoom: DEFAULT_ZOOM,
        center: [MAP_DEFAULT_LON, MAP_DEFAULT_LAT],
    });

    let location = useLocation();
    let history = useHistory();

    const request: MutableRefObject<RequestHelper> = useRef(new RequestHelper({ requestConfigs: ZONES_REQUESTS }));

    let { notificationDispatch } = useContext(NotificationCenterContext) || {};

    useEffect(() => {
        getZones();

        let mapOptions = {
            zoom: DEFAULT_ZOOM,
            center: getCenter(),
        };

        setMapOptions(mapOptions);

        return () => {
            request.current.abort();
        };
    }, []);

    useEffect(() => {
        let searchParams = new URLSearchParams(location.search);
        let selectedZone = searchParams.get(SELECTED_ZONE_CGI);
        setSelectedZone(selectedZone);
    }, [location]);

    useEffect(() => {
        let isCreateZoneInfoFilled = !!creatingAreasCoordinates.length && !!createZoneForm.zoneName;
        setIsCreateZoneInfoFilled(isCreateZoneInfoFilled);
    }, [creatingAreasCoordinates, createZoneForm]);

    const getZones = () => {
        setIsLoading(true);
        setError(null);
        request.current
            .exec(REQUESTS.GET_ZONES)
            .finally(() => {
                setIsLoading(false);
            })
            .then((response) => {
                let zones = response?.objects?.company_zones ?? [];
                setZones(zones);
            })
            .catch(setError);
    };

    const createZone = () => {
        setIsCreateZoneMode(true);
        setSelectedZone(null);
    };

    const stopCreateZone = () => {
        setIsCreateZoneMode(false);
        setCreatingAreasCoordinates([]);
        setCreateZoneForm({ zoneName: '' });
    };

    const onCreateZone = () => {
        setIsCreatingZone(true);
        setCreatingZoneError(null);

        let object = creatingAreasCoordinates.map((creatingAreasCoordinate) => {
            return {
                area_title: createZoneForm.zoneName,
                area_type: 'zone',
                area_details: {
                    group: createZoneForm.zoneName,
                },

                area_coords: creatingAreasCoordinate.map((coords) => coords.join(' ')).join(' '),
            };
        });

        request.current
            .exec(REQUESTS.ADD_ZONE, { body: { object } })
            .finally(() => {
                setIsCreatingZone(false);
            })
            .then(() => {
                setTimeout(() => {
                    stopCreateZone();
                    getZones();
                }, BACKEND_CASH_REFRESH_TIMEOUT);
            })
            .catch((error) => {
                setCreatingZoneError(error);

                notificationDispatch(
                    addNotification({
                        title: i18n('Error while creating geofence'),
                        iconType: NotificationIconType.ERROR,
                    }),
                );
            });
    };

    const onFinishCreatingCoordinates = (areasCoordinates: [number, number][][]) => {
        setCreatingAreasCoordinates(areasCoordinates);
    };

    const onCreateZoneFormChange = ({ zoneName }: { zoneName: string }) => {
        setCreateZoneForm({ zoneName });
    };

    const onRemoveZone = () => {
        getZones();
    };

    const onZoneClick = (zoneName: string) => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(SELECTED_ZONE_CGI, zoneName);
        history.push(`${location.pathname}?${searchParams}`);
    };

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Signal Center')}
                    tabs={<SignalsHeaderTabs />}
                />
            }
            filters={
                <ZonesSidebar
                    isLoading={isLoading}
                    error={error}
                    zones={zones}
                    selectedZone={selectedZone}
                    stopCreateZone={stopCreateZone}
                    isCreateZoneMode={isCreateZoneMode}
                    getZones={getZones}
                    createZone={createZone}
                    onCreateZone={onCreateZone}
                    isCreatingZone={isCreatingZone}
                    creatingZoneError={creatingZoneError}
                    isCreateZoneInfoFilled={isCreateZoneInfoFilled}
                    onCreateZoneFormChange={onCreateZoneFormChange}
                    onRemoveZone={onRemoveZone}
                />
            }
        >
            <ZoneEditor
                options={{
                    zoom: mapOptions.zoom,
                    center: mapOptions.center,
                }}
                isCreateZoneMode={isCreateZoneMode}
                areas={
                    selectedZone
                        ? zones?.[selectedZone]
                        : zones
                        ? Object.values(zones).reduce((allAreas, zoneAreas) => {
                              allAreas.push(...zoneAreas);

                              return allAreas;
                          }, [])
                        : []
                }
                onZoneClick={onZoneClick}
                onFinishCreatingCoordinates={onFinishCreatingCoordinates}
            />
        </SectionLayout>
    );
};

export default Zones;
