import React, { useEffect, useState } from 'react';

import { i18n } from 'components/ui/Buttons/AreasButton/AreasWidget/index.i18n';

import Checked from 'components/ui/Buttons/AreasButton/AreasWidget/checked.component.svg';
import Unchecked from 'components/ui/Buttons/AreasButton/AreasWidget/unchecked.component.svg';

import style from 'components/ui/Buttons/AreasButton/AreasWidget/index.css';

export enum Areas {
    FORBIDDEN_TO_RIDE = 'forbidden_to_ride',
    FORBIDDEN_TO_PARK = 'forbidden_to_park',
}

interface IWidgetItem {
    id: Areas;
    title: string;
    checked: boolean;
}

interface IAreasWidgetProps {
    checked: { [key: string]: boolean };
    onCheck: (value: { id: Areas; checked: boolean }[]) => void;
}

export const AreasWidget = React.forwardRef(
    (props: IAreasWidgetProps, ref: React.MutableRefObject<HTMLDivElement | null>) => {
        let { onCheck: onCheckProps, checked: checkedProps } = props;

        const [widgetItems, setWidgetItems] = useState<IWidgetItem[]>([]);

        useEffect(() => {
            let widgetItems = [
                {
                    id: Areas.FORBIDDEN_TO_RIDE,
                    title: i18n('Forbidden to ride'),
                    checked: checkedProps[Areas.FORBIDDEN_TO_RIDE] ?? false,
                },

                {
                    id: Areas.FORBIDDEN_TO_PARK,
                    title: i18n('Forbidden to park'),
                    checked: checkedProps[Areas.FORBIDDEN_TO_PARK] ?? false,
                },
            ];

            setWidgetItems(widgetItems);
        }, [checkedProps]);

        const onCheck = (id: Areas) => {
            let widgetItemsCopy = widgetItems.map((widgetItem) => Object.assign({}, widgetItem));
            let checkedItem = widgetItemsCopy.find((widgetItem) => widgetItem.id === id);
            if (checkedItem) {
                checkedItem.checked = !checkedItem.checked;

                onCheckProps(
                    widgetItemsCopy.map((widgetItem) => {
                        let { id, checked } = widgetItem;

                        return { id, checked };
                    }),
                );
            }
        };

        return (
            <div
                className={style.areas_widget}
                ref={ref}
            >
                {widgetItems.map((widgetItem) => {
                    let { id, title, checked } = widgetItem;

                    return (
                        <AreasWidgetItem
                            key={id}
                            id={id}
                            title={title}
                            checked={checked}
                            onCheck={onCheck}
                        />
                    );
                })}
            </div>
        );
    },
);

interface IAreasWidgetItemProps {
    id: string;
    title: string;
    checked: boolean;
    onCheck: (id: Areas) => void;
}

const AreasWidgetItem = (props: IAreasWidgetItemProps) => {
    let { title, checked, onCheck, id } = props;

    return (
        <div
            className={style.areas_widget_item}
            onClick={onCheck.bind(null, id)}
        >
            <div className={style.title}>{title}</div>
            <div className={style.check_icon}>{checked ? <Checked /> : <Unchecked />}</div>
        </div>
    );
};
