import React, { useContext } from 'react';

import { EMPTY_DATA } from 'constants/constants';

import GlobalTooltipContext from 'contexts/GlobalTooltipContext';

import { useTooltipMouseHandler } from 'hooks/useTooltipMouseHandler';

import { MetrikaLabel } from 'shared/consts/MetrikaLabel';
import { metrikaLableClick } from 'shared/helpers/metrikaLabelClick/metrikaLableClick';
import { Spin } from 'shared/ui/Spin/Spin';

import { ICommonButtonProps } from 'components/ui/Buttons/Button';
import { Direction } from 'components/ui/DescriptionIcon';

import style from 'components/ui/Buttons/ControlButton/index.css';

export enum ButtonSize {
    L = 'l',
    M = 'm',
    S = 's',
}

interface IControlButtonProps extends ICommonButtonProps {
    size?: ButtonSize;
    isLoading?: boolean;
    secondary?: boolean;
    critical?: boolean;
    leftAlign?: boolean;
    tertiary?: boolean; //same size, but without background color and with border
    shadow?: boolean; //same size, but without background color and with shadow
    fullWidth?: boolean; //big paddings and fullwidth,
    metrikaLabel?: MetrikaLabel;
    description?: string | null;
    className?: string;
}

const ControlButton = (props: IControlButtonProps) => {
    let {
        size = ButtonSize.L,
        title,
        onClick,
        disabled,
        fullWidth,
        critical,
        leftAlign,
        tertiary,
        shadow,
        isLoading,
        secondary,
        description,
        metrikaLabel,
        className,
    } = props;

    const buttonRef = React.useRef<HTMLButtonElement | null>(null);
    useTooltipMouseHandler(buttonRef);
    const { setShow, setTooltipOptions } = useContext(GlobalTooltipContext);

    const onMouseEnter = () => {
        if (description) {
            setShow(true);
            setTooltipOptions({
                title: description,
                direction: Direction.TOP,
                small: true,
            });
        }
    };

    const onWrapClick = () => {
        if (metrikaLabel) {
            metrikaLableClick(metrikaLabel);
        }
        onClick();
    };

    return (
        <button
            disabled={disabled ?? isLoading ?? false}
            className={
                `${style.control_button}` +
                ` ${className ?? ''}` +
                ` ${style[size]}` +
                ` ${secondary ? style.secondary : ''}` +
                ` ${critical ? style.critical : ''}` +
                ` ${leftAlign ? style.left_align : ''}` +
                ` ${tertiary ? style.tertiary : ''}` +
                ` ${shadow ? style.shadow : ''}` +
                ` ${fullWidth ? style.full_width : ''}` +
                ` ${disabled ? style.disabled : ''}`
            }
            ref={buttonRef}
            onMouseEnter={onMouseEnter}
            onClick={onWrapClick}
        >
            {isLoading ? <Spin /> : title || EMPTY_DATA}
        </button>
    );
};

export default ControlButton;
