import * as React from 'react';

import GlobalTooltipContext from 'contexts/GlobalTooltipContext';

import { useTooltipMouseHandler } from 'hooks/useTooltipMouseHandler';

import { StackOrientation } from 'components/ui/Buttons/StackedButtons/types';
import { Direction } from 'components/ui/DescriptionIcon';

import styles from 'components/ui/Buttons/StackedButtons/index.css';

interface IStackedButton {
    id: string;
    icon: React.ReactElement;
    onClick?: () => void;
    disabled?: boolean;
    tooltipContent?: string;
    active?: boolean;
}

export interface IStackedButtonsProps {
    orientation?: StackOrientation;
    buttons: IStackedButton[];
}

export const StackedButtons = (props: IStackedButtonsProps) => {
    let { buttons, orientation = StackOrientation.HORIZONTAL } = props;

    return (
        <div className={`${styles.stackedButtons} ${styles[orientation]}`}>
            {buttons.map((button) => {
                return (
                    <StackedButton
                        button={button}
                        key={button.id}
                    />
                );
            })}
        </div>
    );
};

const StackedButton = (props: { button: IStackedButton }) => {
    let { icon, onClick, disabled, active, tooltipContent } = props.button;
    let [isActive, setActive] = React.useState(active);
    let buttonRef = React.useRef<HTMLButtonElement>(null);
    let { setShow, setTooltipOptions } = React.useContext(GlobalTooltipContext);
    let onMouseEnter = () => {
        if (tooltipContent) {
            setShow(true);
            setTooltipOptions({
                title: tooltipContent,
                direction: Direction.TOP,
                small: true,
            });
        }
    };

    React.useEffect(() => {
        /*
            No React onMouseEnter to fire mouseenter (show tooltip) in button only
            AFTER mouseleave (hide tooltip) on another button was fired
            More: https://github.com/facebook/react/issues/10471
        */
        buttonRef.current?.addEventListener('mouseenter', onMouseEnter);

        return () => {
            buttonRef.current?.removeEventListener('mouseenter', onMouseEnter);
        };
    }, [buttonRef.current]);

    useTooltipMouseHandler(buttonRef);

    return (
        <button
            ref={buttonRef}
            className={`${styles.button} ${disabled ? styles.disabled : ''}`}
            onClick={() => {
                if (!disabled && onClick) {
                    setActive((prev) => !prev);
                    onClick?.();
                }
            }}
        >
            <span className={`${styles.icon} ${!isActive ? styles.buttonNotPressed : ''}`}>{icon}</span>
        </button>
    );
};
