import React, { useEffect, useState } from 'react';
import ReactDOM from 'react-dom';

import CloseIcon from 'components/ui/Close';
import ImageError from 'components/ui/Image/ImageError';
import OverlayContainer, { OverlayIndex } from 'components/ui/OverlayContainer';
import Header2 from 'components/ui/Text/Header2';
import Subhead1 from 'components/ui/Text/Subhead1';

import ArrowIcon from 'components/ui/Gallery/arrow.component.svg';

import styles from 'components/ui/Gallery/index.css';

interface IGalleryProps {
    onClose: () => void;
    url: string;
    curIndex: number;
    maxIndex: number;

    next?: () => void;
    prev?: () => void;
    label?: string;
}

const Gallery = (props: IGalleryProps) => {
    let { next, prev, onClose, url } = props;
    let [error, setError] = useState<any>(null);
    const onKeyUp = (event) => {
        if (['ArrowRight', 'Space'].includes(event.code)) {
            next?.();
        }

        if (['ArrowLeft', 'Backspace'].includes(event.code)) {
            prev?.();
        }

        if (event.code === 'Escape') {
            onClose();
        }
    };

    useEffect(() => {
        window.addEventListener('keyup', onKeyUp);

        return () => {
            window.removeEventListener('keyup', onKeyUp);
            setError(null);
        };
    }, [url]);

    useEffect(() => {
        let img = new Image();
        img.onerror = (error) => {
            setError(error);
        };
        img.src = url;
    }, [url]);

    return ReactDOM.createPortal(
        <div className={styles.gallery_container}>
            <OverlayContainer
                onClose={props.onClose}
                zIndex={OverlayIndex.gallery}
            />
            <div className={styles.gallery_content}>
                <div className={styles.header}>
                    <div>
                        <Header2 className={styles.text_color}>{props.label}</Header2>
                    </div>
                    <div className={styles.counter}>
                        <Subhead1 className={styles.text_color}>
                            {props.curIndex + 1} / {props.maxIndex}
                        </Subhead1>
                    </div>
                    <div
                        className={styles.close}
                        onClick={props.onClose}
                    >
                        <CloseIcon />
                    </div>
                </div>
                {!error ? (
                    <div
                        className={styles.img}
                        style={{
                            backgroundImage: `url(${props.url})`,
                        }}
                    ></div>
                ) : (
                    <div className={styles.error}>
                        <ImageError error={error} />
                    </div>
                )}
                <div
                    className={styles.arrow_left}
                    onClick={prev}
                >
                    <ArrowIcon />
                </div>
                <div
                    className={styles.arrow_right}
                    onClick={next}
                >
                    <ArrowIcon />
                </div>
            </div>
        </div>,
        document.querySelector('#main') as Element,
    );
};

export default Gallery;
