import React, { CSSProperties } from 'react';

import { EMPTY_DATA, ENTER_KEY_CODE } from 'constants/constants';

import style from 'components/ui/RadioButton/index.css';

const MIN_VALUES_LENGTH = 2;

export interface IRadioButtonValues {
    value: string | number;
    placeholder: string;
    disabled?: boolean;
}

interface IRadioButtonProps {
    values: IRadioButtonValues[];
    selectedValue?: string | number | null;
    onChange: (value: string | number | null) => void;
    secondary?: boolean;
    mini?: boolean;
    fullWidth?: boolean;
}

interface IRadioButtonState {
    selectedValue: string | number | null;
}

export class RadioButton extends React.Component<IRadioButtonProps, IRadioButtonState> {
    state: IRadioButtonState = {
        selectedValue: null,
    };

    componentDidMount() {
        let { values, selectedValue } = this.props;

        if (values.length < MIN_VALUES_LENGTH) {
            let mockValues = new Array(MIN_VALUES_LENGTH - values.length).fill(true).map((_, index) => {
                return {
                    value: index.toString(),
                    placeholder: EMPTY_DATA,
                    disabled: true,
                };
            });

            values.splice(values.length, MIN_VALUES_LENGTH - values.length, ...mockValues);
        }

        if (
            selectedValue !== null &&
            selectedValue !== undefined &&
            values.find((value) => value.value === selectedValue)
        ) {
            this.setState({ selectedValue: selectedValue });
        } else if (values.length) {
            this.setState({ selectedValue: values?.[0]?.value }, () => {
                this.props.onChange(this.state.selectedValue);
            });
        }
    }

    componentDidUpdate(prevProps: Readonly<IRadioButtonProps>, prevState: Readonly<IRadioButtonState>): void {
        if (this.props.selectedValue !== prevProps.selectedValue) {
            if (this.props.selectedValue !== null && this.props.selectedValue !== undefined) {
                this.setState({ selectedValue: this.props.selectedValue });
            } else {
                this.setState({ selectedValue: null });
            }
        }
    }

    onValueClick(selectedValue: string | number) {
        let { onChange } = this.props;
        this.setState({ selectedValue }, () => {
            onChange(this.state.selectedValue);
        });
    }

    onKeyUp(value: string | number, event: any) {
        const code = event.keyCode ? event.keyCode : event.which;
        if (code == ENTER_KEY_CODE) {
            this.onValueClick(value);
        }
    }

    render() {
        let { values, secondary, fullWidth, mini } = this.props;
        let { selectedValue } = this.state;

        let _style = fullWidth ? { width: `calc(100% / ${values.length})` } : ({} as CSSProperties);

        return (
            <div
                className={`${style.radio_button_group} ${mini ? style.mini : ''} ${fullWidth ? style.full_width : ''}`}
            >
                <div className={style.values_container}>
                    {values.map((valuesItem, index, arr) => {
                        let isValuePrevToSelected = arr[index + 1] ? arr[index + 1].value === selectedValue : false;
                        let isValueNextToSelected = arr[index - 1] ? arr[index - 1].value === selectedValue : false;
                        let { value, placeholder, disabled } = valuesItem;

                        return (
                            <div
                                key={value}
                                tabIndex={0}
                                style={_style}
                                className={
                                    `${style.radio_button}` +
                                    ` ${selectedValue !== null && selectedValue === value ? style.selected : ''}` +
                                    ` ${secondary ? style.secondary : ''}` +
                                    ` ${mini ? style.radio_button_mini : ''}` +
                                    ` ${disabled ? style.disabled : ''}` +
                                    ` ${isValuePrevToSelected ? style.prev_to_selected : ''}` +
                                    ` ${isValueNextToSelected ? style.next_to_selected : ''}`
                                }
                                onClick={
                                    !disabled && selectedValue !== value ? this.onValueClick.bind(this, value) : null
                                }
                                onKeyUp={!disabled && selectedValue !== value ? this.onKeyUp.bind(this, value) : null}
                            >
                                {placeholder}
                                <div className={style.focus_container} />
                            </div>
                        );
                    })}
                </div>
            </div>
        );
    }
}
