import * as React from 'react';
import { useState } from 'react';
import { useLocation } from 'react-router-dom';

import { downloadXLS } from 'features/DownloadXLS/helpers/downloadXLS/downloadXLS';

import { ReportSection } from 'entities/Report/consts/ReportSection';

import { Path } from 'shared/consts/Path';
import { ShimmerContext } from 'shared/ui/GlobalShimmer/GlobalShimmer';

import GroupModal from 'components/Cars/CarsGroups/GroupModal';
import { downloadCarTableXLSX } from 'components/Cars/CarsTable/downloadCarTableXLSX';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { downloadRidesTableXLSX } from 'components/Rides/RidesTable/downloadRidesTableXLSX';
import { downloadSignalsTableXLSX } from 'components/Signals/SignalsTable/downloadSignalsTableXLSX';
import { ButtonPopup } from 'components/ui/Buttons/ButtonPopup';
import ControlButton, { ButtonSize } from 'components/ui/Buttons/ControlButton';
import { MetaControlOptionItem } from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlOption';
import { SAVE_REPORT_REQUESTS } from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/request';

import { RequestHelper } from '../../../../../../../request-helper/src';

import { i18n } from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/index.i18n';

import Download from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/download.component.svg';
import Group from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/group.component.svg';
import Save from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/save.component.svg';
import XLSXIcon from 'components/ui/SectionLayout/SectionFiltersSidebar/MetaControlButton/MetaControlSave/xlsx.component.svg';

interface IMetaControlSaveProps {
    disabled?: boolean;
    onClickProps: () => void;
    section: ReportSection | undefined;
    abortGettingData?: () => void;
    getData?: (options: any) => void;
    xlsxLabel?: string;
    tagsFilter?: string | null;
}

const saveTableXLSX = async (request, section, abortGettingData, getData, location?): Promise<Error | void> => {
    let table;
    switch (section) {
        case ReportSection.CARS:
            table = await downloadCarTableXLSX(request, abortGettingData, getData, location);
            break;
        case ReportSection.RIDES:
            table = await downloadRidesTableXLSX(request, getData);
            break;
        case ReportSection.SIGNALS:
            table = await downloadSignalsTableXLSX(getData);
            break;
    }

    if (table) {
        downloadXLS(table);
    }
};

export const MetaControlSave = (props: IMetaControlSaveProps) => {
    let { disabled, onClickProps, section, abortGettingData, getData, xlsxLabel, tagsFilter } = props;

    const [isCreateGroupModalOpen, setIsCreateGroupModalOpen] = useState<boolean>(false);

    const { showShimmer } = React.useContext(ShimmerContext);
    let { notificationDispatch } = React.useContext(NotificationCenterContext) || {};
    let request = React.useRef(new RequestHelper({ requestConfigs: SAVE_REPORT_REQUESTS }));
    let location = useLocation();

    let onClickXlsx = () => {
        showShimmer?.(true);

        saveTableXLSX(request?.current, section, abortGettingData, getData, location)
            .finally(() => {
                showShimmer?.(false);
            })
            .catch(() => {
                notificationDispatch(addNotification(notificationOptions));
            });
    };

    let notificationOptions = {
        title: i18n('Failed to load XLSX'),
        iconType: NotificationIconType.ERROR,
        delay: 7000,
        button: {
            label: i18n('Try again'),
            onClick: onClickXlsx,
        },
    };

    const openCreateGroupModal = () => {
        setIsCreateGroupModalOpen(true);
    };

    const closeCreateGroupModal = (success?: boolean) => {
        setIsCreateGroupModalOpen(false);
        if (success) {
            notificationDispatch(
                addNotification({
                    title: i18n('Group created'),
                    link: {
                        label: i18n('Go to Groups section'),
                        to: Path.CARS_GROUPS,
                    },
                }),
            );
        }
    };

    let menuOptions = [
        {
            text: 'XLSX',
            icon: XLSXIcon,
            additionalText: xlsxLabel,
            onClick: onClickXlsx,
        },

        { text: i18n('Save as report'), icon: Save, onClick: onClickProps },
        ...(tagsFilter !== undefined && tagsFilter !== null
            ? [{ text: i18n('Add to group'), icon: Group, onClick: openCreateGroupModal }]
            : []),
    ];

    return (
        <>
            <ButtonPopup
                button={(togglePopup) => (
                    <ControlButton
                        fullWidth
                        size={ButtonSize.M}
                        tertiary
                        leftAlign
                        disabled={disabled}
                        title={
                            <>
                                <Download />
                                {i18n('Save')}
                            </>
                        }
                        onClick={togglePopup as () => void}
                    />
                )}
                contentContainer={(togglePopup) => (
                    <>
                        {menuOptions.map(({ text, icon, onClick, additionalText }, index) => {
                            return (
                                <MetaControlOptionItem
                                    key={index}
                                    Icon={icon}
                                    title={text}
                                    additionalText={additionalText}
                                    onClick={() => {
                                        onClick();
                                        togglePopup();
                                    }}
                                />
                            );
                        })}
                    </>
                )}
            />

            {isCreateGroupModalOpen ? (
                <GroupModal
                    onClose={closeCreateGroupModal}
                    type="car"
                    conditions={tagsFilter ?? ''}
                />
            ) : null}
        </>
    );
};
