import React from 'react';

import { ILangProps } from 'components/Cars/CarsTable';
import { IOptionInfo } from 'components/ui/Select/index';

import { i18n } from 'components/ui/Select/OptionsList/index.i18n';

import style from 'components/ui/Select/OptionsList/index.css';

interface IOptionsListProps extends ILangProps {
    options: IOptionInfo[];
    selectOption: (option: IOptionInfo) => void;
    selectedValue?: string;
    searchValue?: string;
}

export const OptionsList = ({ options: optionsProps, selectOption, selectedValue, searchValue }: IOptionsListProps) => {
    // const [searchValue, setSearchValue] = useState<string>('');
    // const [focusedOptionIndex, setFocusedOptionIndex] = useState<null | number>(null);
    // const optionsList: MutableRefObject<HTMLDivElement | null> = useRef(null);

    const filterItems = (): IOptionInfo[] => {
        let options: IOptionInfo[] = optionsProps;
        if (searchValue) {
            options = options.filter((option) => {
                return Object.values(option).some((value) => {
                    return value?.toString()?.toLowerCase()?.includes(searchValue?.toLowerCase());
                });
            });
        }

        return options.sort((option1, option2) => {
            let displayText1 = option1.text ?? option1.value ?? '';
            let displayText2 = option2.text ?? option2.value ?? '';

            return displayText1.localeCompare(displayText2);
        });
    };

    let filteredOptions = filterItems();

    return (
        <div className={style.options}>
            {filteredOptions?.length ? (
                filteredOptions.map((filteredOption) => {
                    return (
                        <OptionsItem
                            option={filteredOption}
                            key={filteredOption.value}
                            searchValue={searchValue}
                            selectedValue={selectedValue}
                            selectOption={selectOption}
                        />
                    );
                })
            ) : (
                <div className={style.no_options}>{i18n('No options')}</div>
            )}
        </div>
    );
};

interface IOptionsItemProps {
    option: IOptionInfo;
    selectOption: (option: IOptionInfo) => void;
    selectedValue?: string;
    searchValue?: string;
}

const OptionsItem = (props: IOptionsItemProps) => {
    let { option, selectOption, selectedValue, searchValue } = props;
    let isSelected = option?.value === selectedValue;

    let titleDisplayText: string = ['string', 'number', 'bigint', 'boolean'].includes(typeof option.text)
        ? option.text?.toString() || option.value?.toString()
        : option.text ?? option.value.toString();

    let getSearchMatch = (searchValue: string | undefined, value: string) => {
        if (searchValue) {
            let startIndex = value?.toLowerCase().indexOf(searchValue?.toLowerCase());

            if (startIndex === -1) {
                return {
                    beforeMatch: value,
                };
            }

            return {
                beforeMatch: value.slice(0, startIndex),
                match: value.slice(startIndex, startIndex + searchValue.length),
                afterMatch: value.slice(startIndex + searchValue.length, value.length),
            };
        } else {
            return value;
        }
    };

    return (
        <div
            className={`${style.option} ${isSelected ? style.selected : ''}`}
            title={titleDisplayText}
            onClick={selectOption.bind(null, option)}
        >
            {Object.entries(getSearchMatch(searchValue, titleDisplayText)).map(([key, value]: [string, string]) => {
                return key === 'match' ? <span className={style.match}>{value}</span> : value;
            })}
        </div>
    );
};
