import React, { useRef, useState } from 'react';

import { useOutsideClickHandler } from 'hooks/useOutsideClickHandler';

import { OptionsList } from 'components/ui/Select/OptionsList';

import { i18n } from 'components/ui/Select/index.i18n';

import ClearIcon from 'components/ui/Select/clear_icon.component.svg';
import SearchIcon from 'components/ui/Select/search_icon.component.svg';

import style from 'components/ui/Select/index.css';

export interface IOptionInfo {
    readonly value: string;
    text?: string;
}

export interface ISelectProps {
    options: IOptionInfo[];
    placeholder: string;
    onSelect: (value: string | null) => void;
    selectedValue?: string;
    disabled?: boolean;
    withSearch?: boolean;
    isClosable?: boolean;
}

const Select = ({
    options = [],
    placeholder,
    onSelect: onSelectProps,
    selectedValue,
    disabled,
    withSearch,
    isClosable,
}: ISelectProps) => {
    const [isOptionsListShow, setIsOptionsListShow] = useState<boolean>(false);
    const [isFocused, setIsFocused] = useState<boolean>(false);
    const [searchValue, setSearchValue] = useState<string | undefined>(undefined);

    const selectRef = useRef(null);

    useOutsideClickHandler(selectRef, () => {
        setIsFocused(false);
        setIsOptionsListShow(false);
    });

    const onSelectClickHandler = (): void => {
        if (!disabled) {
            setIsFocused(true);
            setIsOptionsListShow(isClosable ? !isOptionsListShow : true);
        }
    };

    const onSelectOption = (pickedOption: IOptionInfo): void => {
        setIsOptionsListShow(isClosable ? !isOptionsListShow : false);
        onSelectProps(pickedOption.value);
    };

    const selectedOption = selectedValue ? options.find((option) => option.value === selectedValue) ?? null : null;

    return (
        <div
            tabIndex={0}
            className={
                `${withSearch ? style.withSearch : ''} ${style.select} ${isFocused ? style.focused_select : ''} ` +
                ` ${isOptionsListShow ? style.options_show : ''} ${disabled ? style.disabled : ''} `
            }
            onClick={onSelectClickHandler}
            ref={selectRef}
        >
            {!withSearch ? (
                <div className={style.placeholder}>
                    {selectedOption ? (
                        options && Array.isArray(options) && options.length ? (
                            <div
                                title={selectedOption?.text}
                                key={selectedOption?.value}
                                className={style.current_value}
                            >
                                <span>{selectedOption?.text}</span>
                            </div>
                        ) : null
                    ) : (
                        placeholder ?? i18n('Select')
                    )}
                </div>
            ) : (
                <>
                    <SearchIcon />
                    <input
                        className={`${style.searchInput} ${isFocused ? style.isFocused : ''}`}
                        placeholder={isFocused ? '' : placeholder}
                        value={searchValue ?? ''}
                        onChange={(e) => {
                            setSearchValue(e.target.value);
                        }}
                    />

                    <span onClick={() => setSearchValue(undefined)}>
                        <ClearIcon />
                    </span>
                </>
            )}

            {isOptionsListShow ? (
                <OptionsList
                    options={options}
                    selectedValue={selectedValue}
                    selectOption={onSelectOption}
                    searchValue={searchValue}
                />
            ) : null}
        </div>
    );
};

export default Select;
