import React, { useEffect, useState } from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { useSingleDoubleClick } from 'hooks/useSingleDoubleClick';

import { ORDER_DESC_CGI, ORDER_FIELD_CGI } from 'shared/consts/order';
import { Spin } from 'shared/ui/Spin/Spin';

import DescriptionIcon, { Direction } from 'components/ui/DescriptionIcon';
import ErrorReloadLabel from 'components/ui/ErrorLabel/ErrorReloadLabel';
import { IHeaderInfo, ITableGetMore, ITableRowData, ITableSorting } from 'components/ui/Table/types';

import { i18n } from 'components/ui/Table/index.i18n';

import SortArrowIcon from 'components/ui/Table/sort_arrow.component.svg';

import style from 'components/ui/Table/index.css';

interface ITableProps {
    header: IHeaderInfo[];
    tableData: ITableRowData[];
    isLoading?: boolean;
    error?: Error | null;
    reloadFunction?: () => void;
    getMore?: ITableGetMore | null;
    onSortChange?: (key?: string, sort_desc?: boolean) => void;
}

const Table = (props: ITableProps) => {
    let {
        header: headerProps,
        tableData,
        isLoading,
        error,
        reloadFunction,
        onSortChange,
        getMore = { canGetMore: false },
    } = props;
    let { canGetMore, onGetMoreClick, isMoreLoading, getMoreTitle } = getMore ?? {};
    const [sortDesc, setSortDesc] = useState<ITableSorting>({});

    let location = useLocation();
    let history = useHistory();

    let data = tableData ?? [];
    let header = headerProps ?? [];

    useEffect(() => {
        let [sortDescNew] = getNewSortDesc();
        setSortDesc(sortDescNew);
    }, [header]);

    useEffect(() => {
        let [sortDescNew, orderField, orderDesc] = getNewSortDesc();
        setSortDesc(sortDescNew);

        if (onSortChange) {
            if (orderField) {
                onSortChange(orderField, orderDesc === 'true');
            } else {
                onSortChange();
            }
        }
    }, [location]);

    const getNewSortDesc = (): [ITableSorting, string | null, string | null] => {
        let urlSearchParams = new URLSearchParams(location?.search);
        let orderField = urlSearchParams.get(ORDER_FIELD_CGI);
        let orderDesc = urlSearchParams.get(ORDER_DESC_CGI);

        let sortDescNew: ITableSorting = header
            .filter((headerItem) => headerItem.sorting)
            .reduce((result: ITableSorting, headerItem) => {
                let { key } = headerItem;
                result[key] = {
                    enabled: key === orderField,
                    desc: key === orderField ? orderDesc === 'true' : true,
                };

                return result;
            }, {});

        return [sortDescNew, orderField, orderDesc];
    };

    const onHeadItemClick = (headerItem: IHeaderInfo) => {
        let { key, sorting } = headerItem;

        if (sorting) {
            let urlSearchParams = new URLSearchParams(location?.search);
            let desc = sortDesc?.[key]?.enabled ? (!sortDesc?.[key]?.desc).toString() : 'true';
            urlSearchParams.set(ORDER_FIELD_CGI, key);
            urlSearchParams.set(ORDER_DESC_CGI, desc);

            history.push(`${location.pathname}?${urlSearchParams}`);
        }
    };

    const isDrawLoadingContainer =
        isLoading || isMoreLoading || error || data.length === 0 || header.length === 0 || canGetMore;

    return (
        <>
            <div className={style.table_rows_container}>
                {!isLoading && !error ? (
                    <table className={style.table}>
                        <thead>
                            <tr>
                                {header.map((headerItem) => {
                                    let { key, displayName, description, descriptionDirection, sorting } = headerItem;

                                    return (
                                        <th
                                            key={key}
                                            className={sorting ? style.sorting : ''}
                                        >
                                            <div
                                                className={style.head_cell}
                                                onClick={onHeadItemClick.bind(null, headerItem)}
                                            >
                                                <span className={style.head_name}>{displayName}</span>
                                                {sorting ? (
                                                    <SortArrowIcon
                                                        className={
                                                            `${style.sort_arrow_icon}` +
                                                            ` ${sortDesc?.[key]?.enabled ? style.enabled : ''}` +
                                                            ` ${!sortDesc?.[key]?.desc ? style.desc_down : ''}`
                                                        }
                                                    />
                                                ) : null}
                                                {description ? (
                                                    <DescriptionIcon
                                                        description={description}
                                                        direction={descriptionDirection ?? Direction.RIGHT}
                                                    />
                                                ) : null}
                                            </div>
                                        </th>
                                    );
                                })}
                            </tr>
                        </thead>
                        <tbody className={isDrawLoadingContainer ? style.with_loading_container : ''}>
                            {data.map((tableDataItem, index) => {
                                return (
                                    <TableRow
                                        header={header}
                                        key={`${tableDataItem.meta.key}_${index}`}
                                        tableDataItem={tableDataItem}
                                    />
                                );
                            })}
                        </tbody>
                    </table>
                ) : null}
            </div>
            {isDrawLoadingContainer ? (
                <div
                    className={`${style.controls_container} ${isLoading || isMoreLoading ? style.loading : ''}`}
                    onClick={!isLoading && !isMoreLoading && !error && canGetMore ? onGetMoreClick?.bind(null) : null}
                >
                    {isLoading || isMoreLoading ? (
                        <Spin />
                    ) : error ? (
                        <ErrorReloadLabel
                            title={i18n('Error while loading data')}
                            reloadFunction={reloadFunction}
                        />
                    ) : canGetMore ? (
                        <span className={style.more_button}>{getMoreTitle ?? i18n('Show more')}</span>
                    ) : data.length === 0 || header.length === 0 ? (
                        <span className={style.no_data_message}>{i18n('No information')}</span>
                    ) : null}
                </div>
            ) : null}
        </>
    );
};

const TableRow = ({ header, tableDataItem }: { header: IHeaderInfo[]; tableDataItem: ITableRowData }) => {
    let { data, meta } = tableDataItem;
    let { className } = meta;

    let onSingleClick = () => {
        if (!window.getSelection()?.toString()) {
            meta?.onClick?.();
        }
    };

    let onClick = useSingleDoubleClick(onSingleClick, (e) => e.stopPropagation());

    return (
        <tr
            className={className ?? ''}
            onClick={onClick}
        >
            {header.map((headerItem, index) => {
                let { key } = headerItem;

                return <td key={`${key}_${index}`}>{data?.[key] ?? null}</td>;
            })}
        </tr>
    );
};

export default Table;
