import React from 'react';

import { CarModelSchema } from 'entities/Car/types/CarModelSchema';
import { CarStatusSchema } from 'entities/Car/types/CarStatusSchema';

import { mergeNextPageNumber, useFetchInfinite } from 'shared/hooks/useFetchInifinite/useFetchInfinite';

interface UseCarListServerData {
    tags_filter?: string;
    sensors?: string;
    traits?: string;
    status?: string;

    page_number?: number;
    page_size?: number;
    // todo: rewrite type more carefully
}

interface UseCarListServerRes {
    page_number: number;
    can_get_more_pages: boolean;

    models: Record<string, CarModelSchema>;

    cars: Array<{
        id: string;
        number: string;
        vin: string;
        model_id: string;
        status: string;
    }>;

    statuses: CarStatusSchema[];
}

export type UseCarListData = Omit<UseCarListServerData, 'page_number'>;

export type UseCarListRes = Pick<UseCarListServerRes, 'cars' | 'models' | 'statuses'>;

export function useCarList(data: UseCarListData, timeout?: number) {
    const options = React.useMemo(() => {
        return {
            timeout,
            mergeNext: mergeNextPageNumber,
            mergeData(prevData: Optional<UseCarListRes>, resData: UseCarListServerRes): Optional<UseCarListRes> {
                if (!prevData) {
                    return {
                        cars: resData.cars,
                        statuses: resData.statuses,
                        models: resData.models,
                    };
                } else {
                    return {
                        cars: [...prevData.cars, ...resData.cars],
                        models: { ...prevData.models, ...resData.models },
                        statuses: resData.statuses.reduce(
                            (memo, item) => {
                                if (!memo.find(({ id }) => id === item.id)) {
                                    memo.push(item);
                                }

                                return memo;
                            },
                            [...prevData.statuses],
                        ),
                    };
                }
            },
        };
    }, [timeout]);

    return useFetchInfinite<UseCarListServerData, UseCarListRes, UseCarListServerRes>(
        '/api/leasing/car/list',
        data,
        options,
    );
}
