import { ONE_MILLISECOND } from 'constants/constants';

import { formatCarServerDate } from 'entities/Car/helpers/formatCarServerDate/formatCarServerDate';
import { parseCarOfferConfig } from 'entities/Car/helpers/parseCarOfferConfig/parseCarOfferConfig';
import { CarModelSchema } from 'entities/Car/types/CarModelSchema';
import { CarOfferConfigSchema } from 'entities/Car/types/CarOfferConfigSchema';
import { CarOfferFormSchema } from 'entities/Car/types/CarOfferFormSchema';
import { CarOfferSchema } from 'entities/Car/types/CarOfferSchema';
import { CarOfferSessionSchema } from 'entities/Car/types/CarOfferSessionSchema';
import { CarOfferStageSchema } from 'entities/Car/types/CarOfferStageSchema';
import { CarOfferStatusSchema } from 'entities/Car/types/CarOfferStatusSchema';
import { CarSearchServerSchema } from 'entities/Car/types/CarSearchServerSchema';

import { formatTime } from 'shared/helpers/formatTime/formatTime';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { useFetchRequest } from 'shared/hooks/useFetchRequest/useFetchRequest';

export interface UseCarOfferReq {
    session_id: string;
    traits: 'ReportOffer';
}

export interface UseCarOfferServerRes {
    models: Record<string, CarModelSchema>;
    sessions: [
        {
            car: CarSearchServerSchema;
            stage?: CarOfferStageSchema;
            segment: {
                offer: CarOfferSchema;
                actual_since?: number;
                actual_until?: number;
                diff?: {
                    start_fuel_level: number;
                    finish_fuel_level?: number;
                    current_fuel_level: number;

                    start_mileage: number;
                    finish_mileage?: number;
                    current_mileage: number;
                    mileage: number;
                };
            };
        },
    ];
}

export interface UseCarOfferRes {
    offerConfig: CarOfferConfigSchema;
    offer: Omit<CarOfferFormSchema, 'user'>;
    session: CarOfferSessionSchema;
    raw: CarOfferSchema;
}

const fetchOptions = {
    parser(res: UseCarOfferServerRes): UseCarOfferRes {
        const models = res.models;
        const session = res.sessions[0];
        const segment = session.segment;
        const offer = segment.offer;
        const diff = segment.diff;

        const since = formatCarServerDate(offer.since);
        const until = formatCarServerDate(offer.until);

        return {
            raw: offer,
            offerConfig: parseCarOfferConfig(offer),
            offer: {
                car: {
                    ...session.car,
                    model_name: models[session.car.model_id]?.name,
                },
                offer_options: offer.offer_options,

                delivery_location: {
                    location: offer.delivery_location,
                    name: offer.delivery_location_name,
                },
                return_location: { location: offer.return_location, name: offer.return_location_name },

                since: getDateStartDay(since),
                since_time: formatTime(since),
                until: getDateStartDay(until),
                until_time: formatTime(until),

                insurance_type: offer.insurance_type,
                comment: offer.comment,
                currency: offer.currency,
                total_payment: offer.total_payment,
                deposit: offer.deposit,
                limit_km_per_day: offer.limit_km_per_day,
                overrun_cost_per_km: offer.overrun_cost_per_km,
                status: offer.status.toLowerCase() as CarOfferStatusSchema,
            },
            session: {
                stage: session.stage,

                since,
                until,
                duration:
                    ((segment.actual_until || offer.until) - (segment.actual_since || offer.since)) * ONE_MILLISECOND,
                actual_since: segment.actual_since ? formatCarServerDate(segment.actual_since) : undefined,
                actual_until: segment.actual_until ? formatCarServerDate(segment.actual_until) : undefined,

                start_fuel_level: diff?.start_fuel_level,
                finish_fuel_level: diff?.finish_fuel_level,
                current_fuel_level: diff?.current_fuel_level,

                start_mileage: diff?.start_mileage,
                finish_mileage: diff?.finish_mileage,
                current_mileage: diff?.current_mileage,
                mileage: diff?.mileage, // delta between start and current || finish
                mileage_limit: offer?.mileage_limit,
            },
        };
    },
};

export function useCarOffer(offerId: string) {
    return useFetchRequest<UseCarOfferReq, UseCarOfferRes, UseCarOfferServerRes>(
        '/api/leasing/sessions/history2',
        {
            session_id: offerId,
            traits: 'ReportOffer',
        },
        fetchOptions,
    );
}
