import React from 'react';

import { ONE_MILLISECOND } from 'constants/constants';

import { formatCarServerDate } from 'entities/Car/helpers/formatCarServerDate/formatCarServerDate';
import { CarOfferPreviewSchema } from 'entities/Car/types/CarOfferPreviewSchema';
import { CarOfferStatusSchema } from 'entities/Car/types/CarOfferStatusSchema';
import { UserStatus } from 'entities/User/consts/UserStatus';

import { useFetchRequest } from 'shared/hooks/useFetchRequest/useFetchRequest';

export interface UseCarsOffersTimetableOptions {
    since: Date;
    until: Date;
}

export interface UseCarsOffersTimetableData {
    since: number;
    until: number;
}

export interface UseCarsOffersTimetableServerRes {
    offers_timetable: Record<
        string,
        Array<{
            status: CarOfferStatusSchema;
            tag_id: string;
            offer_id: string;
            user_id: string;
            since: number;
            until: number;
            actual_since?: number;
            actual_until?: number;
        }>
    >;
    users: Record<
        string,
        {
            first_name: string;
            last_name: string;
            p_name: string;
            phone: string;
            email?: string;
            status: UserStatus;
        }
    >;
    cars: Record<
        string,
        {
            model: string;
            number: string;
        }
    >;
}

export type UseCarsOffersTimetableRes = Record<string, Array<CarOfferPreviewSchema>>;

const fetchOptions = {
    parser(res: UseCarsOffersTimetableServerRes) {
        return Object.keys(res.offers_timetable).reduce((memo, carId) => {
            const list = res.offers_timetable[carId];

            memo[carId] = list.map(({ user_id, ...item }) => {
                let data: CarOfferPreviewSchema = {
                    ...item,
                    since: formatCarServerDate(item.since),
                    until: formatCarServerDate(item.until),
                    actual_since: item.actual_since ? formatCarServerDate(item.actual_since) : undefined,
                    actual_until: item.actual_until ? formatCarServerDate(item.actual_until) : undefined,
                };

                const user = res.users[user_id];

                if (user) {
                    data.user = {
                        id: user_id,
                        first_name: user.first_name,
                        last_name: user.last_name,
                        status: user.status,
                        phone_number: user.phone,
                        email: user.email,
                    };
                }

                return data;
            });

            return memo;
        }, {} as UseCarsOffersTimetableRes);
    },
};

export function useCarsOffersTimetable(options: UseCarsOffersTimetableOptions) {
    const data = React.useMemo<UseCarsOffersTimetableData>(() => {
        return {
            since: options.since.getTime() / ONE_MILLISECOND,
            until: options.until.getTime() / ONE_MILLISECOND,
        };
    }, [options]);

    return useFetchRequest<UseCarsOffersTimetableData, UseCarsOffersTimetableRes, UseCarsOffersTimetableServerRes>(
        '/api/leasing/rental/timetable',
        data,
        fetchOptions,
    );
}
